package io.intino.sezzet.engine;

import io.intino.sezzet.SetStore;
import io.intino.sezzet.model.graph.InstantIterator;
import io.intino.sezzet.model.graph.SezzetGraph;
import io.intino.sezzet.operators.*;
import io.intino.sezzet.setql.SetQL;
import io.intino.sezzet.setql.graph.Expression;
import io.intino.sezzet.setql.graph.Expression.Predicate;
import io.intino.sezzet.setql.graph.Expression.Predicate.*;
import io.intino.sezzet.setql.graph.SetqlGraph;
import io.intino.sezzet.setql.graph.rules.Operator;
import io.intino.sezzet.setql.graph.rules.Scale;

import java.io.File;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import static io.intino.sezzet.setql.graph.rules.Operator.AND;
import static io.intino.sezzet.setql.graph.rules.Operator.OR;
import static io.intino.sezzet.setql.graph.rules.Scale.*;
import static java.time.temporal.ChronoUnit.MINUTES;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;

public class SezzetEngine {
	private SetStore setStore;
	private final SezzetGraph sezzetGraph;
	private Instant reference;

	public SezzetEngine(SetStore setStore, SezzetGraph sezzetGraph) {
		this.setStore = setStore;
		this.sezzetGraph = sezzetGraph;
	}

	public File process(String expression) {
		SetqlGraph graph = SetQL.parseAndResolve(expression, new Locale("es", "ES"), sezzetGraph);
		if (graph == null) return null;
		return process(graph.expression(), Instant.now());
	}

	File process(Expression expression, Instant reference) {
		this.reference = reference;
		return processExpression(expression);
	}

	private File processExpression(Expression expression) {
		return setStore.storeSegment(Instant.now(), expression.name$(), doProcessExpression(expression));
	}

	private SetStream doProcessExpression(Expression expression) {
		List<SetStream> streams = expression.operandList().stream().map(this::process).collect(toList());
		SetStream result = streams.get(0);
		for (int i = 1; i < streams.size(); i++)
			result = doOperation(asList(result, streams.get(i)), expression.operand(i).operator());
		return result;
	}

	private SetStream process(Expression.Operand operand) {
		return operand.i$(Predicate.class) ? process(operand.a$(Predicate.class)) :
				doProcessExpression(operand.a$(Expression.InnerExpression.class).expression());
	}

	private SetStream process(Predicate predicate) {
		List<SetStream> maps = predicate.argumentList().stream().map(a -> process(a, predicate)).collect(toList());
		Operator operator = predicate.operator();
		return doOperation(maps, operator);
	}

	private SetStream doOperation(List<SetStream> maps, Operator operator) {
		return maps.size() == 1 ? maps.get(0) :
				operator == OR ? processOR(maps) :
						operator == AND ? processAnd(maps) :
								processDiff(maps);
	}

	private SetStream processOR(List<SetStream> list) {
		return new Union(list);
	}

	private SetStream processAnd(List<SetStream> list) {
		return new Intersection(list);
	}

	private SetStream processDiff(List<SetStream> list) {
		return new Difference(list);
	}

	private Instant present() {
		return reference == null ? Instant.now() : reference;
	}

	private SetStream process(Argument argument, Predicate predicate) {
		List<File> filesToRead = new ArrayList<>();
		for (Instant instant : new InstantIterator(from(predicate), to(predicate), setStore.scale()))
			filesToRead.addAll(files(instant, predicate, argument));
		return new Union(filesToRead.stream().map(FileReader::new).collect(toList()),
				predicate.frequency() != null ? predicate.frequency().lowBound() : 0,
				predicate.frequency() != null ? predicate.frequency().highBound() : Integer.MAX_VALUE,
				predicate.recency() != null ? recencyIndex(predicate) : -1
		);
	}

	private Instant from(Predicate predicate) {
		Period period = predicate.period();
		return period.i$(FromNow.class) ?
				present().minus(minutes(period.a$(FromNow.class)), MINUTES) :
				period.a$(TimeRange.class).from();
	}

	private Instant to(Predicate predicate) {
		Period period = predicate.period();
		return period.i$(FromNow.class) ?
				present() :
				period.a$(TimeRange.class).to();
	}

	private List<File> files(Instant instant, Predicate condition, Argument argument) {
		return argument.i$(SingleValue.class) ? files(instant, condition.property(), argument.a$(SingleValue.class).value()) :
				files(instant, condition.property(), argument.a$(Range.class).lowBound(), argument.a$(Range.class).highBound());
	}

	private List<File> files(Instant instant, String tank, String argument) {
		return setStore.setsOf(tank, argument, instant).stream()
				.map(v -> setStore.fileOf(tank, v, instant))
				.collect(toList());
	}

	private List<File> files(Instant instant, String tank, int lowBound, int highBound) {
		return setStore.setsOf(tank, lowBound, highBound, instant).stream()
				.map(v -> setStore.fileOf(tank, v, instant))
				.collect(toList());
	}

	private long minutes(FromNow fromNow) {
		return amountOf(fromNow.amount(), fromNow.scale());
	}

	private long minutes(Recency recency) {
		return amountOf(recency.amount(), recency.scale());
	}

	private int amountOf(int amount, Scale scale) {
		return scale == Hour ? amount * 60 :
				scale == Day ? amount * 24 * 60 :
						scale == Month ? amount * 30 * 24 * 60 :
								amount * 365 * 30 * 24 * 60;
	}

	private int recencyIndex(Predicate predicate) {
		Instant from = reference;
		Instant to = reference.minus(minutes(predicate.recency()), ChronoUnit.MINUTES);
		int count = 1;
		while (from.isAfter(to)) {
			count++;
			from = setStore.scale().minus(reference);
		}
		return count;
	}

}
