package io.intino.sezzet.engine;

import io.intino.sezzet.operators.FileReader;
import io.intino.sezzet.operators.LongStream;
import io.intino.sezzet.operators.SetStream;

import java.util.ArrayList;
import java.util.List;

public class Featurizer implements FeatureStream {

	private FeatureRecord current = null;
	private FeatureRecord next = null;
	private SetStream stream;
	private List<SetStreamStruct> featureStreams = new ArrayList<>();

	private Featurizer(SetStream stream) {
		this.stream = stream;
	}

	private Featurizer(long[] ids) {
		this.stream = new LongStream(ids);
	}

	public static FeaturizerBuilder create(SetStream stream) {
		return new FeaturizerBuilder(new Featurizer(stream));
	}

	public static FeaturizerBuilder create(long[] ids) {
		return new FeaturizerBuilder(new Featurizer(ids));
	}

	@Override
	public FeatureRecord current() {
		return current;
	}

	@Override
	public FeatureRecord next() {
		if (current == next) hasNext();
		current = next;
		return current;
	}

	@Override
	public boolean hasNext() {
		if (current != next) return true;
		if (!stream.hasNext()) {
			next = null;
			return false;
		}
		long id = stream.next();
		advanceStreamsUntil(id);
		next = new FeatureRecord(id);
		for (SetStreamStruct featureStream : featureStreams)
			if (featureStream.stream.current() == id) next.add(featureStream.feature, featureStream.value);
		return true;
	}


	private void advanceStreamsUntil(long value) {
		for (SetStreamStruct featureStream : featureStreams)
			while (featureStream.stream.current() < value && featureStream.stream.hasNext())
				featureStream.stream.next();
	}

	public static class FeaturizerBuilder {

		private Featurizer featurizer;

		FeaturizerBuilder(Featurizer featurizer) {
			this.featurizer = featurizer;
		}

		public FeaturizerBuilder add(SetStream stream, String feature, String value) {
			featurizer.featureStreams.add(new SetStreamStruct(stream, feature, value));
			return this;
		}

		public FeaturizerBuilder add(FileReader stream) {
			featurizer.featureStreams.add(new SetStreamStruct(stream, stream.feature(), stream.value()));
			return this;
		}

		public Featurizer commit() {
			return featurizer;
		}
	}

	private static class SetStreamStruct {

		private final SetStream stream;
		private final String feature;
		private final String value;

		SetStreamStruct(SetStream stream, String feature, String value) {
			this.stream = stream;
			this.feature = feature;
			this.value = value;
		}

	}
}
