package io.intino.sezzet.editor.box.displays;

import io.intino.konos.alexandria.ui.displays.AlexandriaStamp;
import io.intino.sezzet.editor.box.displays.notifiers.SezzetEditorNotifier;
import io.intino.sezzet.editor.box.schemas.ChangeParameters;
import io.intino.sezzet.editor.box.schemas.Setup;
import io.intino.sezzet.editor.box.schemas.SuggestParameters;
import io.intino.sezzet.editor.box.schemas.ValidationItem;
import io.intino.sezzet.model.graph.Category;
import io.intino.sezzet.model.graph.Feature;
import io.intino.sezzet.model.graph.Group;
import io.intino.sezzet.model.graph.SezzetGraph;
import io.intino.sezzet.setql.SetqlChecker;
import io.intino.sezzet.setql.SetqlParser;
import io.intino.sezzet.setql.exceptions.SemanticException;
import io.intino.sezzet.setql.exceptions.SetqlError;
import io.intino.sezzet.setql.exceptions.SyntaxException;
import io.intino.sezzet.setql.graph.Expression;
import io.intino.sezzet.setql.graph.SetqlGraph;
import io.intino.sezzet.setql.graph.rules.Operator;
import io.intino.tara.magritte.Graph;
import io.intino.tara.magritte.Layer;

import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

public class SezzetEditor extends AlexandriaStamp<SezzetEditorNotifier> {
	private List<Consumer<ChangeContext>> listeners = new ArrayList<>();
	private SetqlGraph setql;
	private SezzetGraph sezzetGraph;
	private final List<Feature> features;
	private String expression;
	private boolean readonly = false;

	public SezzetEditor(String dsl) {
		this.setql = new Graph().loadStashes("Setql").as(SetqlGraph.class);
		this.sezzetGraph = new Graph().loadStashes("Sezzet", dsl).as(SezzetGraph.class);
		this.features = sezzetGraph.allFeatures();
	}

	public String expression() {
		return this.expression;
	}

	public SezzetEditor expression(String expression) {
		this.expression = expression;
		return this;
	}

	public SezzetEditor readonly(boolean value) {
		this.readonly = value;
		return this;
	}

	public SezzetEditor addListener(Consumer<ChangeContext> listener) {
		listeners.add(listener);
		return this;
	}

	@Override
	protected void init() {
		super.init();
		notifier.render(new Setup().
				readonly(readonly).
				features(features()).
				invalidLabel(sezzetGraph.invalidLabel().value()).
				undefinedLabel(sezzetGraph.undefinedLabel().value()));
	}

	@Override
	public void refresh() {
		notifier.setExpression(expression);
		validate();
	}

	public void onChange(ChangeParameters params) {
		this.expression = params.expression();
		SetqlGraph graph = validate();
		listeners.forEach(l -> l.accept(changeContext(params, graph)));
	}

	private ChangeContext changeContext(ChangeParameters params, SetqlGraph graph) {
		String line = lineOf(params);
		return new ChangeContext() {
			public boolean existsOperator() {
				return !line.startsWith("*");
			}

			public String operand() {
				return line;
			}

			public Expression expression() {
				return graph == null ? null : graph.expression();
			}

			public String rawExpression() {
				return params.expression();
			}

			public int lineNumber() {
				return params.cursor().row();
			}
		};
	}

	private String lineOf(ChangeParameters parameters) {
		String[] split = parameters.expression().split("\n");
		return split[parameters.cursor().row()].trim();
	}

	private SetqlGraph validate() {
		SetqlGraph graph = graph();
		List<ValidationItem> items = new ArrayList<>();
		if (expression != null && !expression.isEmpty()) validate(items, graph);
		notifier.validationResult(items);
		return graph;
	}

	private void validate(List<ValidationItem> items, SetqlGraph graph) {
		try {
			String language = session().browser().languageFromMetadata();
			Locale locale = language.equals("es") || language.equals("mx") ? new Locale("es", "ES") : Locale.ENGLISH;
			new SetqlParser(expression, locale).check(graph);
			new SetqlChecker(sezzetGraph, locale).check(graph);
		} catch (SyntaxException e) {
			for (SetqlError error : e.errors())
				items.add(new ValidationItem().line(error.line()).message(((SyntaxException.SyntaxError) error).lineMessage()));
		} catch (SemanticException e) {
			for (SetqlError error : e.errors())
				items.add(new ValidationItem().line(error.line()).message(error.message()));
		}
	}

	public void onSuggest(SuggestParameters params) {
		String line = params.line();
		if (isInArgumentContext(line, params.cursor())) {
			notifier.suggestion(arguments(featureIn(line, params.cursor()).toLowerCase()));
		} else if (startsWithOperator(line) && !line.contains("::") || line.indexOf("::") > params.cursor())
			notifier.suggestion(features(params.line(), params.cursor()));
		else if (line.contains("::")) {
			final String temporal = line.substring(line.indexOf("::"));
			List<String> options = new ArrayList<>();
			if (!temporal.contains("period")) options.add("period");
			if (!temporal.contains("recency")) options.add("recency");
			if (!temporal.contains("frequency")) options.add("frequency");
			notifier.suggestion(options);
		}
	}

	private List<String> features() {
		return features.stream().map(Feature::label).collect(toList());
	}


	private List<String> features(String line, int cursor) {
		String feature = featureIn(line, cursor);
		String prefix = feature.contains(".") ? feature.substring(0, feature.lastIndexOf(".")) : feature;
		String suffix = feature.contains(".") ? feature.substring(feature.lastIndexOf(".") + 1) : feature;
		String[] names = prefix.split("\\.");
		if (names.length == 1) {
			return names[0].equals(sezzetGraph.subject().value()) ?
					map(sezzetGraph.groupList(), ".", suffix) :
					singletonList(sezzetGraph.subject().value() + ".");
		}
		Group group = findGroup(Arrays.copyOfRange(names, 1, names.length));
		if (group == null) return Collections.emptyList();
		List<String> values = map(group.groupList(), ".", suffix);
		values.addAll(map(group.featureList(), "", suffix));
		return values;
	}

	private Group findGroup(String[] names) {
		Group currentGroup = null;
		for (String name : names) {
			if (currentGroup == null) {
				currentGroup = findGroup(sezzetGraph.groupList(), name);
				if (currentGroup == null) return null;
			} else currentGroup = findGroup(currentGroup.groupList(), name);
		}
		return currentGroup;
	}

	private List<String> map(List<? extends Layer> layers, String prefix, String suffix) {
		return layers.stream().map(g -> g.name$() + prefix).filter(g2 -> g2.startsWith(suffix)).collect(Collectors.toList());
	}

	private Group findGroup(List<Group> groupList, String name) {
		for (Group g : groupList) if (g.name$().equals(name)) return g;
		return null;
	}

	private boolean startsWithOperator(String line) {
		char start = line.trim().isEmpty() ? 0 : line.trim().charAt(0);
		return start != 0 && (start == '*' || Operator.fromText(start + "") != null) && hasIndent(line);
	}

	private boolean hasIndent(String line) {
		return (line.trim().length() > 1 && line.trim().charAt(1) == '\t') || (line.length() == 2 && line.charAt(1) == '\t');
	}

	private List<String> arguments(String featureName) {
		Feature feature = features.stream().filter(f -> f.label().equalsIgnoreCase(featureName)).findFirst().orElse(null);
		if (feature == null) return Collections.emptyList();
		List<String> arguments = new ArrayList<>();
		if (feature.isEnumerate())
			arguments.addAll(sezzetGraph.valuesOf(feature).stream().map(Category::label).collect(toList()));
		else if (feature.isText()) arguments.addAll(feature.asText().suggestions());
		else if (feature.isNumeric()) arguments.addAll(feature.asNumeric().suggestions());
		if (feature.isAllowEmpty()) arguments.add(sezzetGraph.undefinedLabel().value());
		if (feature.isAllowNaS()) arguments.add(sezzetGraph.invalidLabel().value());
		return arguments;
	}

	private String featureIn(String line, int cursor) {
		String text = line.indexOf(")", cursor) > 0 ? line.substring(0, line.indexOf(")", cursor)) : line;
		text = !text.contains("(") ? text : text.substring(0, text.lastIndexOf("("));
		if (text.contains(" ")) text = text.substring(text.lastIndexOf(" ") + 1);
		return text.substring(text.lastIndexOf("\t") + 1);
	}

	private boolean isInArgumentContext(String line, int cursor) {
		int rightParenthesis = line.indexOf(")", cursor);
		int leftParenthesis = line.indexOf("(");
		return leftParenthesis > 0 && cursor > leftParenthesis && (cursor <= rightParenthesis || !line.contains(")"));
	}

	private SetqlGraph graph() {
		return setql.core$().clone().as(SetqlGraph.class);
	}


	public interface ChangeContext {
		boolean existsOperator();

		String operand();

		Expression expression();

		String rawExpression();

		int lineNumber();
	}
}