package io.intino.builderservice;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.net.URL;
import java.lang.reflect.Type;
import com.google.gson.JsonSerializer;
import com.google.gson.JsonDeserializer;
import io.intino.alexandria.exceptions.*;
import io.intino.alexandria.restaccessor.RequestBuilder;
import io.intino.alexandria.restaccessor.core.RestAccessorNotifier;
import io.intino.alexandria.restaccessor.exceptions.RestfulFailure;

import io.intino.builderservice.schemas.*;

public class QuassarBuilderServiceAccessor {
	private final URL url;
	private final RestAccessorNotifier notifier = new RestAccessorNotifier();
	private int timeoutMillis = 120 * 1_000;
	private io.intino.alexandria.restaccessor.OutBox outBox = null;
	private Map<String, String> additionalHeaders = new HashMap<>();

	public enum OutputResourceOutput {
		Src, Gen, Res, Out, Build
	}

	public QuassarBuilderServiceAccessor(URL url) {
		this.url = url;

	}

	public QuassarBuilderServiceAccessor(URL url, int timeoutMillis) {
		this.url = url;
		this.timeoutMillis = timeoutMillis;

	}

	public QuassarBuilderServiceAccessor(URL url, int timeoutMillis, java.io.File outBoxDirectory, int intervalRetrySeconds) {
		this.url = url;
		this.timeoutMillis = timeoutMillis;
		this.outBox = new io.intino.alexandria.restaccessor.OutBox(outBoxDirectory, intervalRetrySeconds);

	}

	public void addCommonHeader(String name, String value) {
		additionalHeaders.put(name, value);
	}

	public void addRequestSerializer(Type type, JsonSerializer<?> adapter) {
		io.intino.alexandria.restaccessor.adapters.RequestAdapter.addCustomAdapter(type, adapter);
	}

	public void addResponseDeserializer(Type type, JsonDeserializer<?> adapter) {
		io.intino.alexandria.restaccessor.adapters.ResponseAdapter.addCustomAdapter(type, adapter);
	}

	public List<BuilderInfo> getBuilders() throws InternalServerError {
		RequestBuilder builder = new RequestBuilder(this.url).timeOut(this.timeoutMillis);
		additionalHeaders.forEach((k,v) -> builder.headerParameter(k,v));
		RequestBuilder.Request request = builder
			.build(RequestBuilder.Method.GET, "/api" + "/builders");
		try {
			io.intino.alexandria.restaccessor.Response response = request.execute();
			return io.intino.alexandria.restaccessor.adapters.ResponseAdapter.adapt(response.content(), new com.google.gson.reflect.TypeToken<ArrayList<BuilderInfo>>(){}.getType());
		} catch (AlexandriaException e) {

			if (outBox != null) outBox.push(request);
			throw new InternalServerError(e.message());
		}
	}

	public void postBuilders(RegisterBuilder info) throws InternalServerError {
		RequestBuilder builder = new RequestBuilder(this.url).timeOut(this.timeoutMillis);
		additionalHeaders.forEach((k,v) -> builder.headerParameter(k,v));
		RequestBuilder.Request request = builder
			.entityPart("info", info)
			.build(RequestBuilder.Method.POST, "/api" + "/builders");
		try {
			io.intino.alexandria.restaccessor.Response response = request.execute();
		} catch (AlexandriaException e) {

			if (outBox != null) outBox.push(request);
			throw new InternalServerError(e.message());
		}
	}

	public BuilderInfo getBuilderInfo(String imageURL, String registryToken) throws Conflict, InternalServerError {
		RequestBuilder builder = new RequestBuilder(this.url).timeOut(this.timeoutMillis);
		additionalHeaders.forEach((k,v) -> builder.headerParameter(k,v));
		RequestBuilder.Request request = builder
			.queryParameter("imageURL", imageURL)
			.entityPart("registryToken", registryToken)
			.build(RequestBuilder.Method.GET, "/api" + "/builders" + "/info");
		try {
			io.intino.alexandria.restaccessor.Response response = request.execute();
			return io.intino.alexandria.restaccessor.adapters.ResponseAdapter.adapt(response.content(), BuilderInfo.class);
		} catch (AlexandriaException e) {
			if (e instanceof Conflict) throw ((Conflict) e);
			if (outBox != null) outBox.push(request);
			throw new InternalServerError(e.message());
		}
	}

	public String postRunOperation(RunOperationContext runOperationContext, io.intino.alexandria.Resource.InputStreamProvider filesInTar) throws InternalServerError {
		RequestBuilder builder = new RequestBuilder(this.url).timeOut(this.timeoutMillis);
		additionalHeaders.forEach((k,v) -> builder.headerParameter(k,v));
		RequestBuilder.Request request = builder
			.entityPart("runOperationContext", runOperationContext)
			.entityPart(new io.intino.alexandria.Resource("filesInTar", filesInTar))
			.build(RequestBuilder.Method.POST, "/api" + "/operations" + "/run");
		try {
			io.intino.alexandria.restaccessor.Response response = request.execute();
			return io.intino.alexandria.restaccessor.adapters.ResponseAdapter.adapt(response.content(), String.class);
		} catch (AlexandriaException e) {

			if (outBox != null) outBox.push(request);
			throw new InternalServerError(e.message());
		}
	}

	public OperationResult getOperationOutput(String ticket) throws InternalServerError {
		RequestBuilder builder = new RequestBuilder(this.url).timeOut(this.timeoutMillis);
		additionalHeaders.forEach((k,v) -> builder.headerParameter(k,v));
		RequestBuilder.Request request = builder
			.build(RequestBuilder.Method.GET, "/api" + "/operation" + "/" + ticket + "/output");
		try {
			io.intino.alexandria.restaccessor.Response response = request.execute();
			return io.intino.alexandria.restaccessor.adapters.ResponseAdapter.adapt(response.content(), OperationResult.class);
		} catch (AlexandriaException e) {

			if (outBox != null) outBox.push(request);
			throw new InternalServerError(e.message());
		}
	}

	public io.intino.alexandria.Resource getOutputResource(String ticket, OutputResourceOutput output, String excludeFilePattern) throws NotFound, InternalServerError {
		RequestBuilder builder = new RequestBuilder(this.url).timeOut(this.timeoutMillis);
		additionalHeaders.forEach((k,v) -> builder.headerParameter(k,v));
		RequestBuilder.Request request = builder
			.queryParameter("excludeFilePattern", excludeFilePattern)
			.build(RequestBuilder.Method.GET, "/api" + "/operation" + "/" + ticket + "/outputs" + "/" + output);
		try {
			io.intino.alexandria.restaccessor.Response response = request.execute();
			String filename = !response.headers().containsKey("Content-Disposition") ? "filename=content":
				Arrays.stream(response.headers().get("Content-Disposition").split(";")).filter(c-> c.startsWith("filename")).findFirst().orElse(null);
			return new io.intino.alexandria.Resource(filename.split("=")[1], response.contentType(), response.contentAsStream());
		} catch (AlexandriaException e) {
			if (e instanceof NotFound) throw ((NotFound) e);
			if (outBox != null) outBox.push(request);
			throw new InternalServerError(e.message());
		}
	}

	public void listenSubscribeOperation(String ticket, java.util.function.Consumer<String> listener) throws InternalServerError {
		RequestBuilder.Request request = new RequestBuilder(this.url).timeOut(this.timeoutMillis)
			.build(RequestBuilder.Method.POST, "operation/:ticket/messages");
		try {
			this.notifier.listen(listener, request.execute().content().trim());
		} catch (AlexandriaException e) {
			throw new InternalServerError(e.message());
		}
	}

	public void stopListenSubscribeOperation() {
		this.notifier.close();
	}
}