package io.intino.monet.messaging.senders;

import io.intino.alexandria.logger.Logger;
import io.intino.monet.messaging.*;
import io.intino.monet.messaging.pushnotifications.PushNotification;

import java.util.Map;

public class PushSender implements Sender {

    private final Messaging messaging;

    public PushSender(Messaging messaging) {
        this.messaging = messaging;
    }

    @Override
    public boolean send(Notification notification) {
        try {
            if(notification.channel() != Notification.Channel.Push) return false;
            if(!messaging.pushNotifications().service().enabled()) return false;
            if(messaging.pushNotifications().blacklist().isDisabled(notification.type())) return false;
            return sendPushNotification(notification);
        } catch (Exception e) {
            Logger.error(e);
            return false;
        }
    }

    private boolean sendPushNotification(Notification notification) {

        Recipient recipient = notification.recipient();

        String recipientEmail = recipient.email();

        if(recipient.deviceToken() == null) {
            recipient = messaging.recipientsStore().get(recipient.email());
        }

        if(recipient == null || recipient.deviceToken() == null) {
            Logger.warn("Recipient " + recipientEmail + " does not defines a device token. Cannot send push notification");
            return false;
        }

        NotificationTemplate template = messaging.pushNotifications().getPushTemplate(notification.type(), notification.language());
        if(template == null) return false;

        for(String variable : template.properties()) {
            template.set(variable, notification.properties().getOrDefault(variable, ""));
        }

        PushNotification push = new PushNotification(notification.id())
                .recipient(recipient)
                .title(template.subject())
                .body(template.body())
                .ts(notification.ts());

        push.properties().putAll(notification.properties());

        for(Map.Entry<String, Notification.Attachment> attachment : notification.attachments().entrySet()) {
            push.putExtraData(attachment.getKey(), attachment.getValue().value);
        }

        messaging.pushNotifications().service().send(push);

        return true;
    }
}
