package io.intino.monet.messaging.senders;

import io.intino.alexandria.logger.Logger;
import io.intino.monet.messaging.*;
import io.intino.monet.messaging.emails.Email;
import io.intino.monet.messaging.emails.EmailSignature;

import java.io.File;

public class EmailSender implements Sender {

    private final Messaging messaging;

    public EmailSender(Messaging messaging) {
        this.messaging = messaging;
    }

    @Override
    public boolean send(Notification notification) {
        try {
            if(notification.channel() != Notification.Channel.Email) return false;
            if(!messaging.emails().service().enabled()) return false;
            if(messaging.emails().blacklist().isDisabled(notification.type())) return false;
            return sendEmail(notification);
        } catch (Exception e) {
            Logger.error(e);
            return false;
        }
    }

    private boolean sendEmail(Notification notification) {
        NotificationTemplate template = getEmailTemplate(notification.type(), notification.language());

        Recipient recipient = notification.recipient();

        for(String variable : template.properties()) {
            template.set(variable, notification.properties().getOrDefault(variable, ""));
        }

        Email email = new Email(new EmailSignature(notification.id()))
                .recipients().to(recipient)
                .subject(template.subject())
                .body().text(template.body());

        email.properties().putAll(notification.properties());

        if(notification.properties().containsKey("content-type")) {
            email.body().contentType(notification.properties().get("content-type"));
        }

        for(Notification.Attachment attachment : notification.attachments().values()) {
            if(attachment.type == Notification.Attachment.Type.File)
                email.body().attachments().add(new File(attachment.value));
        }

        messaging.emails().service().send(email);

        return true;
    }

    private NotificationTemplate getEmailTemplate(String type, String language) {
        return messaging.emails().getEmailTemplate(type, language);
    }
}
