package io.intino.monet.messaging.pushnotifications;

import com.google.common.collect.Lists;
import io.intino.alexandria.Json;

import java.io.File;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.nio.file.StandardOpenOption.CREATE;
import static java.util.Objects.requireNonNull;

public class PushNotificationStore {

    private final File root;
    private final int maxNotificationsPerUser;
    private final Object mutex = new Object();

    public PushNotificationStore(File root) {
        this(root, 64);
    }

    public PushNotificationStore(File root, int maxNotificationsPerUser) {
        this.root = requireNonNull(root);
        root.mkdirs();
        this.maxNotificationsPerUser = maxNotificationsPerUser;
    }

    public List<PushNotification> loadNotifications(String user) {
        synchronized (mutex) {
            File file = new File(root, user + ".json");
            if(!file.exists() || file.length() == 0) return new ArrayList<>();
            try(Stream<String> lines = Files.lines(file.toPath())) {
                return lines.filter(line -> !line.isBlank())
                        .map(this::deserialize)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList());
            } catch (Exception e) {
                return new ArrayList<>();
            }
        }
    }

    public boolean saveNotifications(String user, List<PushNotification> notifications) {
        synchronized (mutex) {
            File file = new File(root, user + ".json");
            try {
                List<String> json = serialize(notifications);
                if(json.isEmpty()) return true;
                Files.write(file.toPath(), json, CREATE);
                return true;
            } catch (Exception e) {
                return false;
            }
        }
    }

    public void addNotification(PushNotification notification) {
        synchronized (mutex) {
            List<PushNotification> notifications = loadNotifications(notification.recipient().email());
            notifications.add(notification);
            saveNotifications(notification.recipient().email(), notifications);
        }
    }

    private List<String> serialize(List<PushNotification> notifications) {
        return Lists.reverse(notifications).stream()
                .limit(maxNotificationsPerUser)
                .sorted(Comparator.comparing(PushNotification::ts))
                .map(this::serialize)
                .collect(Collectors.toList());
    }

    private String serialize(PushNotification notification) {
        return Json.toString(notification);
    }

    private PushNotification deserialize(String json) {
        return Json.fromString(json, PushNotification.class);
    }

    public File root() {
        return root;
    }

    public int maxNotificationsPerUser() {
        return maxNotificationsPerUser;
    }
}
