package io.intino.monet.messaging.pushnotifications;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import static java.util.Objects.requireNonNull;

public abstract class PushNotificationService {

    private final PushNotificationStore store;
    private final ExecutorService executor;
    private PushNotificationServicePipeline pipeline = new PushNotificationServicePipeline.Default();
    private volatile boolean enabled = true;

    public PushNotificationService(PushNotificationStore store) {
        this.store = requireNonNull(store);
        this.executor = Executors.newSingleThreadExecutor();
    }

    public final void send(PushNotification notification) {
        if(!enabled) return;
        executor.submit(() -> {
            sendNotification(notification);
            saveNotificationToUsersStore(notification);
        });
    }

    protected abstract void sendNotification(PushNotification notification);

    private void saveNotificationToUsersStore(PushNotification notification) {
        store.addNotification(notification);
    }

    public boolean shutdown() {
        try {
            executor.shutdown();
            executor.awaitTermination(1, TimeUnit.HOURS);
            return true;
        } catch (Exception ignored) {
            return false;
        }
    }

    public boolean enabled() {
        return enabled;
    }

    public PushNotificationService enabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    public PushNotificationServicePipeline notificationPipeline() {
        return pipeline;
    }

    public PushNotificationService setNotificationPipeline(PushNotificationServicePipeline pipeline) {
        this.pipeline = pipeline == null ? new PushNotificationServicePipeline.Default() : pipeline;
        return this;
    }

    public PushNotificationStore store() {
        return store;
    }
}
