package io.intino.monet.messaging;

import io.intino.alexandria.logger.Logger;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class NotificationTemplate {

    private static final String PROPERTY_BEGIN = "::";
    private static final String PROPERTY_END = "::";
    private static final String EXTENSION = ".template";
    private static final Pattern PropertiesPattern = Pattern.compile("::[^:\\s]+::");

    public static NotificationTemplate empty() {
        return new NotificationTemplate("", "");
    }

    public static NotificationTemplate load(File root, String type, String language) {
        File file = new File(root, type + "/" + type + "." + language + EXTENSION);
        if(!file.exists()) {
            Logger.error("Template " + file + " not found");
            return null;
        }
        return load(file);
    }

    public static NotificationTemplate load(String text) {
        return parse(text);
    }

    public static NotificationTemplate load(InputStream inputStream) {
        try {
            return parse(new String(inputStream.readAllBytes()));
        } catch (IOException e) {
            Logger.error(e);
            return null;
        }
    }

    public static NotificationTemplate load(File file) {
        try {
            return parse(Files.readString(file.toPath()));
        } catch (Exception e) {
            Logger.error(e);
            return null;
        }
    }

    private static NotificationTemplate parse(String text) {
        String[] content = text.split("\\[body]");
        return new NotificationTemplate(content[0].replace("[subject]", ""), content[1].replace("[body]", ""));
    }

    private String subject;
    private String body;
    private final Set<String> properties;

    public NotificationTemplate(String subject, String body) {
        this.subject = subject;
        this.body = body;
        this.properties = properties(subject, body);
    }

    private Set<String> properties(String subject, String body) {
        Set<String> variables = new LinkedHashSet<>();
        properties(subject, variables);
        properties(body, variables);
        return variables;
    }

    private void properties(String text, Set<String> variables) {
        Matcher matcher = PropertiesPattern.matcher(text);
        while(matcher.find()) {
            variables.add(matcher.group().replace("::", ""));
        }
    }

    public Set<String> properties() {
        return Collections.unmodifiableSet(properties);
    }

    public NotificationTemplate set(String name, String value) {
        subject = subject.replace(property(name), value);
        body = body.replace(property(name), value);
        return this;
    }

    private String property(String name) {
        return PROPERTY_BEGIN + name + PROPERTY_END;
    }

    public String subject() {
        return subject;
    }

    public String body() {
        return body;
    }
}
