package io.intino.monet.messaging.pushnotifications.backends.firebase;

import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.common.collect.Lists;
import com.google.firebase.FirebaseApp;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.messaging.BatchResponse;
import com.google.firebase.messaging.FirebaseMessaging;
import com.google.firebase.messaging.Message;
import com.google.firebase.messaging.SendResponse;
import io.intino.monet.messaging.pushnotifications.PushNotification;
import io.intino.monet.messaging.pushnotifications.PushNotificationServicePipeline;
import io.intino.monet.messaging.pushnotifications.PushNotificationStore;
import io.intino.monet.messaging.pushnotifications.backends.PushNotificationsBackend;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.Proxy;
import java.util.List;

public class FirebasePushNotificationService extends PushNotificationsBackend {

	private final Config config;
	private FirebaseApp firebaseApp;

	public FirebasePushNotificationService(Config config) {
		super("Firebase");
		this.config = config;
	}

	@Override
	protected void doStart() {
		this.firebaseApp = login(config);
	}

	@Override
	public void sendNotification(PushNotification notification) {
		sendMessages(notification, FirebaseMessageBuilder.build(notification));
	}

	public void sendMessages(PushNotification notification, Message message) {
		if(message == null) return;
		try {
			if(!notificationPipeline().onBeforeSend(notification)) return;
			BatchResponse responses = FirebaseMessaging.getInstance(firebaseApp).sendAll(List.of(message));
			notificationPipeline().onAfterSend(notification, getSendResult(responses));
		} catch (Throwable e) {
			notificationPipeline().onError(e);
		}
	}

	private PushNotificationServicePipeline.SendResult getSendResult(BatchResponse responses) {
		PushNotificationServicePipeline.SendResult result = new PushNotificationServicePipeline.SendResult(name());
		for(SendResponse r : responses.getResponses()) {
			PushNotificationServicePipeline.SendResponse sendResponse = new PushNotificationServicePipeline.SendResponse(r.getException());
			sendResponse.info.put("messageId", r.getMessageId());
			sendResponse.info.put("isSuccessful", String.valueOf(r.isSuccessful()));
			result.responses.add(sendResponse);
		}
		return result;
	}

	private FirebaseApp login(Config config) {
		try {
			HttpTransport httpTransport = httpTransport(config.proxy);
			return FirebaseApp.initializeApp(FirebaseOptions.builder()
					.setCredentials(credentials(config.credentialsJson, httpTransport))
					.setConnectTimeout(5000)
					.setReadTimeout(5000)
					.setHttpTransport(httpTransport)
					.build());
		} catch (IOException e) {
			notificationPipeline().onError(e);
			return null;
		}
	}

	private GoogleCredentials credentials(String credentialsJson, HttpTransport httpTransport) throws IOException {
		return GoogleCredentials.fromStream(new ByteArrayInputStream(credentialsJson.getBytes()), () -> httpTransport)
				.createScoped(Lists.newArrayList("https://www.googleapis.com/auth/cloud-platform"));
	}

	private HttpTransport httpTransport(Proxy proxy) {
		NetHttpTransport.Builder builder = new NetHttpTransport.Builder();
		if (proxy != null) builder.setProxy(proxy);
		return builder.build();
	}

	public static class Config {

		private String credentialsJson;
		private Proxy proxy;

		public String credentialsJson() {
			return credentialsJson;
		}

		public Config credentialsJson(String credentialsJson) {
			this.credentialsJson = credentialsJson;
			return this;
		}

		public Proxy proxy() {
			return proxy;
		}

		public Config proxy(Proxy proxy) {
			this.proxy = proxy;
			return this;
		}
	}
}
