package io.intino.monet.messaging.pushnotifications;

import io.intino.monet.messaging.logging.MessagingLog;
import io.intino.monet.messaging.pushnotifications.backends.PushNotificationsBackend;
import io.intino.monet.messaging.pushnotifications.backends.PushNotificationsBackendManager;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import static java.util.Objects.requireNonNull;

public class PushNotificationService {

    private final PushNotificationStore store;
    private final PushNotificationServicePipeline notificationPipeline;
    private final PushNotificationsBackendManager backendManager;
    private ExecutorService executor;
    private volatile boolean enabled = true;

    private PushNotificationService(PushNotificationStore store, PushNotificationsBackendManager backendManager, PushNotificationServicePipeline notificationPipeline) {
        this.store = requireNonNull(store);
        this.backendManager = requireNonNull(backendManager);
        this.notificationPipeline = requireNonNull(notificationPipeline);
    }

    public void start() {
        backendManager.start(notificationPipeline);
        executor = Executors.newSingleThreadExecutor();
    }

    public final void send(PushNotification notification) {
        if(!enabled) return;
        executor.submit(() -> {
            sendNotification(notification);
            saveNotificationToUsersStore(notification);
        });
    }

    private void sendNotification(PushNotification notification) {
        PushNotificationsBackend backend = backendManager.backendOf(notification.recipient().device());
        if(backend == null) {
            notificationPipeline.onBackendNotFound(notification);
            return;
        }
        backend.sendNotification(notification);
    }

    private void saveNotificationToUsersStore(PushNotification notification) {
        store.addNotification(notification);
    }

    public boolean shutdown() {
        try {
            executor.shutdown();
            executor.awaitTermination(1, TimeUnit.HOURS);
            return true;
        } catch (Exception ignored) {
            return false;
        }
    }

    public boolean enabled() {
        return enabled;
    }

    public PushNotificationService enabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    public PushNotificationServicePipeline notificationPipeline() {
        return notificationPipeline;
    }

    public PushNotificationStore store() {
        return store;
    }

    public static class Builder {

        private PushNotificationStore store;
        private Function<List<PushNotificationsBackend>, PushNotificationsBackendManager> backendManager = PushNotificationsBackendManager.Default::new;
        private final List<PushNotificationsBackend> backends = new LinkedList<>();
        private PushNotificationServicePipeline notificationPipeline = new PushNotificationServicePipeline.Default();
        private boolean enabled = true;

        public PushNotificationService build() {
            return new PushNotificationService(store, this.backendManager.apply(backends), notificationPipeline).enabled(enabled);
        }

        public Builder store(PushNotificationStore store) {
            this.store = store;
            return this;
        }

        public Builder addBackend(PushNotificationsBackend backend) {
            backends.add(requireNonNull(backend));
            return this;
        }

        public Builder backendManager(Function<List<PushNotificationsBackend>, PushNotificationsBackendManager> backendManager) {
            this.backendManager = backendManager == null ? PushNotificationsBackendManager.Default::new : backendManager;
            return this;
        }

        public Builder notificationPipeline(PushNotificationServicePipeline notificationPipeline) {
            this.notificationPipeline = notificationPipeline == null ? new PushNotificationServicePipeline.Default() : notificationPipeline;
            return this;
        }

        public Builder enabled(boolean enabled) {
            this.enabled = enabled;
            return this;
        }
    }
}
