package io.intino.monet.messaging.emails;

import io.intino.monet.messaging.emails.store.EmailBlacklist;
import io.intino.monet.messaging.emails.store.EmailStore;
import io.intino.monet.messaging.emails.util.MimeMessageBuilder;

import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.MimeMessage;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import static java.util.Objects.requireNonNull;

public class EmailService {

    public static final String SMTP_HOST = "mail.smtp.hostname";
    public static final String SMTP_USER = "mail.smtp.username";
    public static final String SMTP_PASSWORD = "mail.smtp.password";
    public static final int DEFAULT_MAX_CONNECTION_TRIES = 3;

    private final EmailStore store;
    private final Properties properties;
    private final EmailServicePipeline emailPipeline;
    private ExecutorService executor;
    private Session session;
    private Transport transport;
    private volatile boolean enabled = true;

    private EmailService(EmailStore store, Properties properties, EmailServicePipeline emailPipeline) {
        this.store = requireNonNull(store);
        this.properties = requireNonNull(properties);
        this.emailPipeline = requireNonNull(emailPipeline);
    }

    public void start() {
        this.session = EmailSessionFactory.create(properties);
        this.transport = EmailSessionFactory.getTransport(session);
        this.executor = Executors.newSingleThreadExecutor();
    }

    public void send(Email email) {
        send(Collections.singletonList(email));
    }

    public void send(List<Email> emails) {
        executor.submit(() -> sendEmails(emails));
    }

    private void sendEmails(List<Email> emails) {
        try {
            if(!enabled) return;
            connect();
            for(Email email : emails) {
                sendEmail(email);
            }
        } catch (Throwable e) {
            emailPipeline.onError(e);
        }
    }

    private synchronized void sendEmail(Email email) {
        if(!enabled) return;
        if(!emailPipeline.onBeforeSendEmail(email)) return;

        boolean mustBeSentOncePerDay = mustBeSentOncePerDay(email);

        Set<Reason> reasonsForNotSending = new HashSet<>(2);

        if(!couldSendNow(mustBeSentOncePerDay, email)) reasonsForNotSending.add(Reason.AlreadySent);
        if(!acceptedByRecipient(email)) reasonsForNotSending.add(Reason.NotAcceptedByRecipient);

        final boolean shouldSend = reasonsForNotSending.isEmpty();
        if (shouldSend) {
            if (!sendMessage(buildMimeMessage(email))) return;
            if (mustBeSentOncePerDay) registerEmailSent(email);
        }

        emailPipeline.onAfterSendEmail(email, shouldSend, reasonsForNotSending);
    }

    private boolean sendMessage(MimeMessage message) {
        try {
            transport.sendMessage(message, message.getAllRecipients());
            return true;
        } catch (MessagingException e) {
            emailPipeline.onError(e);
            return false;
        }
    }

    private MimeMessage buildMimeMessage(Email email) {
        return new MimeMessageBuilder()
                .session(session)
                .senderAddress(properties.getProperty("mail.from"))
                .senderName(properties.getProperty("mail.from.name"))
                .email(email)
                .build();
    }

    private boolean acceptedByRecipient(Email email) {
        EmailBlacklist blacklist = store.emailBlacklist(email.recipients().to().email());
        return !blacklist.contains(email.signature());
    }

    private boolean couldSendNow(boolean mustBeSentOncePerDay, Email email) {
        if (mustBeSentOncePerDay) return !wasAlreadySentToday(email);
        return true;
    }

    private boolean wasAlreadySentToday(Email email) {
        return store.emailsSent(email.recipients().to().email()).wasSentToday(email.signature().get());
    }

    private void registerEmailSent(Email email) {
        store.emailsSent(email.recipients().to().email()).put(email.signature().get(), LocalDateTime.now());
    }

    private boolean mustBeSentOncePerDay(Email email) {
        return "true".equals(email.properties().get("oncePerDay"));
    }

    private synchronized void connect() throws Exception {
        Exception error = null;
        int maxConnectionTries = getMaxConnectionTries();
        for (int i = 0; i < maxConnectionTries; i++) {
            error = tryConnectToSmtpServer();
            if (error == null) return;
            emailPipeline.onConnectionError(error);
            sleep(1000 * (i + 1));
        }
        if(error != null) throw error;
    }

    private int getMaxConnectionTries() {
        String tries = properties.getProperty("max.connection.tries");
        if(tries == null) return DEFAULT_MAX_CONNECTION_TRIES;
        try {
            int result = Integer.parseInt(tries);
            return result < 0 ? DEFAULT_MAX_CONNECTION_TRIES : result;
        } catch (Exception e) {
            return DEFAULT_MAX_CONNECTION_TRIES;
        }
    }

    private void sleep(int millis) {
        try {
            Thread.sleep(millis);
        } catch (InterruptedException ignored) {}
    }

    private Exception tryConnectToSmtpServer() {
        try {
            if (transport.isConnected()) return null;
            String host = properties.getProperty(SMTP_HOST);
            String user = properties.getProperty(SMTP_USER);
            String password = properties.getProperty(SMTP_PASSWORD);
            transport.connect(host, user, password);
            return null;
        } catch (Exception e) {
            return e;
        }
    }

    public boolean shutdown() {
        try {
            executor.shutdown();
            executor.awaitTermination(1, TimeUnit.HOURS);
            return true;
        } catch (Exception ignored) {
            return false;
        }
    }

    public boolean enabled() {
        return enabled;
    }

    public EmailService enabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    public EmailStore store() {
        return store;
    }

    public enum Reason {
        NotAcceptedByRecipient, AlreadySent
    }

    public static class Builder {

        private EmailStore store;
        private Properties properties;
        private EmailServicePipeline emailPipeline = new EmailServicePipeline.Default();
        private boolean enabled = true;

        public EmailService build() {
            return new EmailService(store, properties, emailPipeline).enabled(enabled);
        }

        public Builder store(EmailStore store) {
            this.store = store;
            return this;
        }

        public Builder properties(Properties properties) {
            this.properties = properties;
            return this;
        }

        public Builder emailPipeline(EmailServicePipeline emailPipeline) {
            this.emailPipeline = emailPipeline;
            return this;
        }

        public Builder enabled(boolean enabled) {
            this.enabled = enabled;
            return this;
        }
    }
}
