package io.intino.monet.messaging;

import io.intino.alexandria.Json;
import io.intino.monet.messaging.emails.util.JsonPretty;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.nio.file.StandardOpenOption.CREATE;

public class RecipientsStore {

    private final File file;
    private final Map<String, Recipient> recipients;

    public RecipientsStore(File directory) {
        this.file = new File(directory, "recipients.json");
        file.getParentFile().mkdirs();
        recipients = new ConcurrentHashMap<>();
        load();
    }

    public Recipient get(String email) {
        return recipients.get(email);
    }

    public void add(Recipient recipient) {
        recipients.put(recipient.email(), recipient);
    }

    public void remove(String email) {
        recipients.remove(email);
    }

    public boolean exists(String email) {
        return recipients.containsKey(email);
    }

    public List<Recipient> getAll() {
        return new ArrayList<>(recipients.values());
    }

    public boolean save() {
        file.getParentFile().mkdirs();
        synchronized (this) {
            try {
                File tmp = createTmpFile();
                Files.writeString(tmp.toPath(), JsonPretty.toJson(recipients.values()), CREATE);
                Files.move(tmp.toPath(), file.toPath(), REPLACE_EXISTING);
                return true;
            } catch (IOException e) {
                return false;
            }
        }
    }

    public void load() {
        synchronized (this) {
            recipients.clear();
            try {
                Recipient[] recipientsFromFile = file.exists() ? Json.fromString(read(file), Recipient[].class) : new Recipient[0];
                Arrays.stream(recipientsFromFile).forEach(r -> recipients.put(r.email(), r));
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private String read(File file) throws IOException {
        return Files.readString(file.toPath());
    }

    private File createTmpFile() {
        return new File(file.getAbsolutePath() + "." + System.currentTimeMillis() + ".tmp");
    }
}
