package io.intino.monet.engine.edition;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class LocalFormStore implements FormStore {
	private final File folder;
	private final Map<String, Object> data;

	public LocalFormStore(File folder) {
		this.folder = folder;
		this.folder.mkdirs();
		this.data = new HashMap<>();
	}

	public File folder() {
		return folder;
	}

	public void load() throws IOException {
		this.data.clear();
		this.data.putAll(read(target().toPath()));
	}

	public void save() throws IOException {
		target().delete();
		write(target().toPath());
	}

	private Map<String, Object> read(Path path) throws IOException {
		List<String> lines = Files.readAllLines(path);
		return lines.stream().map(this::parse).filter(Objects::nonNull).collect(Collectors.toMap(v->v.key, v->v.value));
	}

	private void write(Path path) throws IOException {
		for (String key : data.keySet())
			Files.write(path, new Variable(key,data.get(key)).serialize(), CREATE, APPEND);
	}

	private Variable parse(String line) {
		return !line.isEmpty() ? new Variable(line.split("\t")) : null;
	}

	@Override
	public List<String> keys() {
		return new ArrayList<>(data.keySet());
	}

	@Override
	public Object get(String name) {
		Object value = data.get(name);
		return value != null ? get(value, file(name)) : null;
	}

	private Object get(Object value, File file) {
		return file != null ? new Object[] {value, file} : value;
	}

	private File file(String name) {
		File file = new File(folder, name);
		return file.exists() ? file : null;
	}

	@Override
	public void put(String name, Object value) {
		if (value != null) add(name, value); else remove(name);
	}

	private void add(String name, Object value) {
		if (value instanceof File) move(name, (File) value);
		else if (value instanceof InputStream) save(name, (InputStream) value);
		else if(value instanceof Object[]) Arrays.stream((Object[]) value).forEach(v -> add(name, v));
		else data.put(name, value);
	}

	private void remove(String name) {
		data.remove(name);
		new File(folder, name).delete();
	}

	private void move(String name, File file) {
		File dest = new File(folder, name);
		dest.getParentFile().mkdirs();
		file.renameTo(dest);
	}

	private void save(String name, InputStream is) {
		try {
			File destination = new File(folder, name);
			destination.getParentFile().mkdirs();
			Files.write(destination.toPath(), is.readAllBytes(), StandardOpenOption.CREATE);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@Override
	public int size() {
		return data.size();
	}

	private File target() {
		return new File(folder, "data.tsv");
	}

	private static class Variable {
		public String key;
		public Object value;

		public Variable(String[] split) {
			this.key = split[0];
			this.value = parse(split[1], split.length >= 3 ? split[2] : "");
		}

		private Object parse(String type, String value) {
			if (type.equalsIgnoreCase("Double")) return Double.parseDouble(value);
			if (type.equalsIgnoreCase("LocalDate")) return LocalDate.parse(value);
			return value;
		}

		public Variable(String key, Object value) {
			this.key = key;
			this.value = value;
		}

		public byte[] serialize() {
			return (key + '\t' + typeOf(value) + '\t' + value + '\n').getBytes();
		}

		private String typeOf(Object value) {
			return value.getClass().getSimpleName();
		}
	}
}
