package io.intino.monet.engine.edition;

import java.io.File;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;

import static io.intino.monet.engine.edition.FormStore.*;

public abstract class Edition {
	protected final String name;
	protected final FormStore store;
	protected final boolean isOptional;
	protected final String conditional;
	protected final File attachment;

	public Edition(String name, FormStore store, Map<String, String> options, Language language) {
		this.name = name;
		this.store = store;
		this.isOptional = isTrue(options.get("optional"));
		this.conditional = options.getOrDefault("conditional", null);
		this.attachment = attachment(options, language);
	}

	public Edition init() {
		checkConditional();
		return this;
	}

	public boolean isOptional() {
		return isOptional;
	}

	public boolean isConditional() {
		return conditional != null;
	}

	public File attachment() {
		return attachment;
	}

	public void skip() {
		if (!isOptional()) throw new EditionException("Can't skip non optional field");
		this.store.put(name, skipped);
	}

	public void enable() {
		if (this.store.get(name) != disabled) return;
		this.store.put(name, null);
	}

	public void disable() {
		this.store.put(name, disabled);
	}

	public String name() {
		return name;
	}

	public Object value() {
		return this.store.get(name);
	}

	public boolean isNull() {
		return value() == null;
	}

	public boolean isEmpty() {
		return isEmpty(value());
	}

	public boolean isSkipped() {
		return Objects.equals(value(), skipped);
	}

	public boolean isDisabled() {
		return Objects.equals(value(), disabled);
	}

	public boolean isHidden() {
		return Objects.equals(value(), hidden);
	}

	 @SuppressWarnings("unchecked")
	 public <T extends Edition> T as(Class<T> aClass) {
		 return aClass.isAssignableFrom(aClass) ? (T) this : null;
	 }

	private boolean isTrue(String optional) {
		return "true".equalsIgnoreCase(optional);
	}

	public static class EditionException extends RuntimeException {

	public EditionException(String message) {
		super(message);
	}
	}

	protected boolean isEmpty(Object value) {
		return value == null || (value instanceof String && ((String) value).length() > 0 && ((String) value).charAt(0) == '\30');
	}

	protected String storeValueOf(String name) {
		Object value = store.get(name);
		if (value == null) return name;
		String result = null;
		if (value instanceof Object[]) result = arrayValueOf((Object[]) value);
		else if (value instanceof String) result = (String) value;
		return result;
	}

	private String arrayValueOf(Object[] value) {
		if (value.length == 0) return null;
		return (String) value[0];
	}

	private void checkConditional() {
		if (!isConditional()) return;
		if (fulfillCondition()) enable();
		else disable();
	}

	private boolean fulfillCondition() {
		String name = conditionalName();
		if (name == null) return false;
		String storeValue = storeValueOf(name);
		if (storeValue == null) return false;
		String value = conditionalValue();
		return value != null && Arrays.asList(value.split("\\|")).contains(storeValue);
	}

	private String conditionalName() {
		if (conditional == null) return null;
		return conditional.split("=")[0];
	}

	private String conditionalValue() {
		if (conditional == null) return null;
		return conditional.split("=")[1];
	}

	private File attachment(Map<String, String> options, Language language) {
		String attachment = options.getOrDefault("attachment." + language, null);
		if (attachment == null || attachment.isEmpty()) return null;
		return new File(options.get("attachment-repository") + "/" + attachment);
	}

}
