package io.intino.monet.engine;

import io.intino.alexandria.logger.Logger;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.*;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.*;

public class OrderTypes {

	private static final Map<String, Record> records = new HashMap<>();
	private static final Map<String, BiConsumer<Record, String>> setters = new HashMap<>() {{
		put("label.en", (r, v) -> r.labelEn = v);
		put("label.es", (r, v) -> r.labelEs = v);
		put("label.pt", (r, v) -> r.labelPt = v);
		put("hint.en", (r, v) -> r.hintEn = v);
		put("hint.es", (r, v) -> r.hintEs = v);
		put("hint.pt", (r, v) -> r.hintPt = v);
		put("code", (r, v) -> r.code = v);
		put("effort", (r, v) -> r.effort = Integer.parseInt(v));
		put("input", (r, v) -> r.input = v);
		put("calculations", (r, v) -> r.calculations = v);
		put("annexes", (r, v) -> r.annexes = v);
		put("target", (r, v) -> r.target = v);
		put("channel", (r, v) -> r.channel = Channel.valueOf(v));
		put("assertion", (r, v) -> {
			String[] split = v.split(":");
			r.assertionCode = split[0];
			r.assertionAttr = split.length == 1 ? Collections.emptyMap() :
					Arrays.stream(split[1].split(",")).map(s -> s.split(" as "))
							.collect(toMap(s -> s[0].trim(), s -> s.length == 1 ? s[0].trim() : s[1].trim(), (v1, v2) -> v1, LinkedHashMap::new));
		});
	}};

	public static void init(File file) {
		try {
			records.clear();
			if (!file.exists()) return;
			Files.readAllLines(file.toPath()).stream()
					.filter(l -> !l.trim().isEmpty())
					.map(l -> l.split("\t", -1))
					.forEach(l -> setters.getOrDefault(l[1], nullSetter()).accept(record(l[0]), l[2]));
			all().forEach(r -> load(file, r));
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private static void load(File file, Record record) {
		File orderTypeFile = new File(file.getParentFile(), "order-types/" + record.code + ".triples");
		Map<String, List<String[]>> triples = Checklist.triples(orderTypeFile);
		record.checklist = Checklist.load(orderTypeFile);
		record.triples = record.checklist.fields.stream().sorted(comparing(e -> e.code)).collect(toMap(e -> e, e -> Checklist.asMap(Checklist.triples(triples.getOrDefault(e.code, Collections.emptyList()))), (a, v) -> a, LinkedHashMap::new));
	}

	private static BiConsumer<Record, String> nullSetter() {
		return (record, s) -> {
		};
	}

	public static Collection<Record> all() {
		return records.values();
	}

	public static Record of(String code) {
		return records.get(code);
	}

	private static Record record(String id) {
		if (!records.containsKey(id)) {
			Record record = new Record();
			record.code = id;
			records.put(id, record);
		}
		return records.get(id);
	}

	public enum Channel {web, app, both}

	public static class Record {
		private String code;
		private String labelEn;
		private String labelEs;
		private String labelPt;
		private String hintEn;
		private String hintEs;
		private String hintPt;
		private String target;
		private int effort;
		private String input;
		private String calculations;
		private String annexes;
		private Channel channel;
		private Checklist checklist = new Checklist();
		private Map<Checklist.Field, Map<String, String>> triples = new HashMap<>();
		private String assertionCode;
		private Map<String, String> assertionAttr;

		public String code() {
			return code;
		}

		public String label(String lang) {
			return lang.equals("en") ? labelEn : lang.equals("es") ? labelEs : labelPt;
		}

		public String hint(String lang) {
			return lang.equals("en") ? hintEn : lang.equals("es") ? hintEs : hintPt;
		}

		public String category() {
			return code.startsWith("P") ? "Preventive" : code.startsWith("A") ? "Administrative" : "Corrective";
		}

		public int effort() {
			return effort;
		}

		public boolean isManual() {
			return input == null || input.isEmpty();
		}

		public List<String> input() {
			if (input == null) return Collections.emptyList();
			return Stream.of(input.split(",")).map(String::trim).filter(s -> !s.isEmpty()).collect(toList());
		}

		public List<String> calculations() {
			if (calculations == null) return Collections.emptyList();
			return Stream.of(calculations.split(",")).map(String::trim).filter(s -> !s.isEmpty()).collect(toList());
		}

		public List<String> annexes() {
			if (annexes == null) return Collections.emptyList();
			return Stream.of(annexes.split(",")).map(String::trim).filter(s -> !s.isEmpty()).collect(toList());
		}

		public String target() {
			return target;
		}

		public Channel channel() {
			return channel;
		}

		public String assertionCode() {
			return assertionCode;
		}

		public Map<String, String> assertionAttrs() {
			return assertionAttr;
		}

		public Checklist checklist() {
			return checklist;
		}

		public Map<Checklist.Field, Map<String, String>> triples() {
			return triples;
		}
	}

	public static class Checklist implements Iterable<Checklist.Field> {
		private final List<Field> fields = new ArrayList<>();

		private static Checklist load(File file) {
			return load(triples(file));
		}

		private static Checklist load(Map<String, List<String[]>> triples) {
			Checklist checklist = new Checklist();
			for (String key : triples.keySet())
				checklist.add(key, triples(triples.get(key)));
			sort(checklist.fields);
			checklist.setSections();
			return checklist;
		}

		private static void sort(List<Field> fields) {
			fields.sort(comparing(f -> f.code));
		}

		private static List<Entry> triples(List<String[]> splits) {
			return splits.stream().map(Entry::new).collect(toList());
		}

		private static Map<String, String> asMap(List<Entry> entries) {
			Map<String, String> map = new HashMap<>(entries.size());
			for (Entry entry : entries) map.put(entry.key, entry.value);
			return map;
		}

		private static Map<String, List<String[]>> triples(File file) {
			try {
				return Files.readAllLines(file.toPath()).stream()
						.filter(s -> !s.isEmpty())
						.map(s -> s.split("\t"))
						.collect(groupingBy(s -> s[0]));
			} catch (IOException ignored) {
				return Collections.emptyMap();
			}
		}

		private void setSections() {
			Field section = null;
			for (Field field : fields) {
				if (field.type == Type.Section) section = field;
				else if (section != null) field.section(section);
			}
		}

		public Field field(String key) {
			return fields.stream().filter(f -> f.code.equalsIgnoreCase(key) || f.name.equalsIgnoreCase(key)).findFirst().orElse(null);
		}

		public List<Field> fields() {
			return fields;
		}

		private void add(String code, List<Entry> entries) {
			add(new Field(code, asMap(entries)));
		}

		private void add(Field field) {
			fields.add(field);
		}

		@Override
		public Iterator<Field> iterator() {
			return fields.iterator();
		}

		public enum Type {
			String, Number, Date, Option, MultiOption, Image, Entity, Package, Note, Marker, Section, Signature, Validation
		}

		public static class Field {
			public final String name;
			public final Type type;
			private final String code;
			private final boolean optional;
			private final String conditional;
			private final List<String> filter;
			private final Map<String, String> entries;
			private Field section;

			public Field(String code, Map<String, String> entries) {
				this.code = code;
				this.name = entries.get("name");
				this.type = Type.valueOf(entries.getOrDefault("type", "String"));
				this.entries = entries;
				this.optional = Boolean.parseBoolean(entries.getOrDefault("optional", "false"));
				this.filter = OrderTypes.listOf(entries.getOrDefault("filter", ""));
				this.conditional = entries.get("conditional");
			}

			public String title(String language) {
				return get("title.", language);
			}

			public String description(String language) {
				return get("description.", language);
			}

			public List<String> values(String language) {
				return Stream.of(get("values.", language).split(";")).map(String::trim).collect(toList());
			}

			public Double valueMin() {
				return entries.containsKey("value-min") ? Double.parseDouble(entries.get("value-min")) : null;
			}

			public Double valueMax() {
				return entries.containsKey("value-max") ? Double.parseDouble(entries.get("value-max")) : null;
			}

			public Double valueDefault() {
				return entries.containsKey("value-default") ? Double.parseDouble(entries.get("value-default")) : 0;
			}

			public String unit() {
				return entries.get("unit");
			}

			public boolean isOptional() {
				return optional;
			}

			public boolean isConditional() {
				return conditional != null;
			}

			public String conditional() {
				return conditional;
			}

			public List<String> filter() {
				return filter;
			}

			public Field section() {
				return section;
			}

			public String get(String attribute, String language) {
				return entries.getOrDefault(attribute + language, "");
			}

			@Override
			public String toString() {
				return "Field{" + "type=" + type + ", name='" + name + '\'' + '}';
			}

			private void section(Field section) {
				this.section = section;
			}
		}

		public static class Entry {
			public final String key;
			public final String value;

			public Entry(String[] split) {
				this(split[1], split[2]);
			}

			public Entry(String key, String value) {
				this.key = key;
				this.value = value;
			}
		}

	}

	private static List<String> listOf(String value) {
		if (value == null) return Collections.emptyList();
		return Stream.of(value.split(","))
				.map(String::trim)
				.filter(s -> !s.isEmpty())
				.collect(Collectors.toList());
	}

}
