package io.intino.monet.engine.edition;

import io.intino.monet.engine.edition.editors.*;

import java.util.List;
import java.util.Set;
import java.util.stream.IntStream;

import static java.util.stream.Collectors.toList;

public class Form {
	public final FormDefinition definition;
	public final FormStore store;
	private Language language;

	public Form(FormDefinition definition, FormStore store) {
		this.definition = definition;
		this.store = store;
		this.language = Language.es;
	}

	public Field start(Language language) {
		this.language = language;
		return field(0);
	}

	public Field field(String name) {
		return field(indexOf(name));
	}

	public Field field(int index) {
		return check(index) ? createField(index, this.definition.at(index)) : null;
	}

	public List<Edition> editions(FieldType... types) {
		return editions(Set.of(types));
	}

	private List<Edition> editions(Set<FieldType> types) {
		return store.keys().stream()
				.map(this::indexOf).filter(i -> i != -1).sorted()
				.filter(i->types.isEmpty() || types.contains(definition.at(i).type))
				.map(this::edition)
				.collect(toList());
	}

	public boolean isCompleted() {
		return sizeOf(store) == definition.size();
	}

	public int progress() {
		if (definition.size() == 0) return 100;
		return sizeOf(store) * 100 / definition.size();
	}

	private int sizeOf(FormStore store) {
		return (int) definition.fields().stream().filter(f -> store.get(f.name) != null).count();
	}

	private boolean check(int index) {
		return 0 <= index && index < definition.size();
	}

	private Field createField(int index, FieldDefinition definition) {
		return new Field() {
			@Override
			public String name() {
				return definition.name;
			}

			@Override
			public String header() {
				return isSection(index) ? "" : sectionOf(index).title();
			}

			@Override
			public String title() {
				return definition.get("title." + language.name());
			}

			@Override
			public String description() {
				return definition.get("description." + language.name());
			}

			public FieldDefinition definition() {
				return definition;
			}

			public FieldType type() {
				return definition.type;
			}

			@Override
			public Edition edition() {
				return Form.this.edition(index);
			}

			@Override
			public boolean hasNext() {
				return nextIndex() < Form.this.definition.size();
			}

			@Override
			public boolean hasPrev() {
				return prevIndex() > 0;
			}

			@Override
			public Field next() {
				Field field = field(nextIndex());
				while (field != null && field.edition().isDisabled()) field = field.next();
				return field;
			}

			@Override
			public Field prev() {
				Field field = field(prevIndex());
				while (field != null && field.edition().isDisabled()) field = field.prev();
				return field;
			}

			private int nextIndex() {
				return isSectionAndSkippedOrDisabled(index) ? nextSection(index + 1) : index + 1;
			}

			private int prevIndex() {
				int result = index - 1;
				while (result >= 0 && (isHidden(result) || isDisabled(result))) result--;
				if (result < 0) return 0;
				return result;
			}

			private boolean isSectionAndSkippedOrDisabled(int index) {
				return isSection(index) && (isSkipped(index) || isDisabled(index));
			}

			private boolean isSkipped(int index) {
				return get(index) == FormStore.skipped;
			}

			private boolean isHidden(int index) {
				return get(index) == FormStore.hidden;
			}

			private boolean isDisabled(int index) {
				return get(index) == FormStore.disabled;
			}

			private Object get(int index) {
				return store.get(nameOf(index));
			}
			
		};
	}

	private String nameOf(int index) {
		return check(index) ? Form.this.definition.at(index).name : null;
	}

	private int indexOf(String name) {
		return IntStream.range(0, definition.size())
				.filter(i -> definition.at(i).name.equals(name))
				.findFirst()
				.orElse(-1);
	}

	private Edition edition(int index) {
		FieldDefinition definition = this.definition.at(index);
		switch (definition.type) {
			case String:
				return new StringEdition(definition.name, store, definition.options, language).init();
			case Date:
				return new DateEdition(definition.name, store, definition.options, language).init();
			case Number:
				return new NumberEdition(definition.name, store, definition.options, language).init();
			case Image:
				return new ImageEdition(definition.name, store, definition.options, language).init();
			case Signature:
				return new SignatureEdition(definition.name, store, definition.options, language).init();
			case Note:
				return new NoteEdition(definition.name, store, definition.options, language).init();
			case Package:
				return new PackageEdition(definition.name, store, definition.options, language).init();
			case Option:
				return new OptionEdition(definition.name, store, definition.options, language).init();
			case MultiOption:
				return new OptionMultipleEdition(definition.name, store, definition.options, language).init();
			case Marker:
			case Section:
				return new SectionEdition(definition.name, store, definition.options, sectionOf(index), language).init();
			case Validation:
				return new ValidationEdition(definition.name, store, definition.options, language).init();
		}
		return null;
	}

	private Section sectionOf(int index) {
		return new Section(definitionsOfCurrentSection(index), language);
	}

	private List<FieldDefinition> definitionsOfCurrentSection(int index) {
		int from = previousSection(index);
		return from >= 0 ? definitions(from, nextSection(index+1)) : List.of();
	}

	private List<FieldDefinition> definitions(int from, int to) {
		return IntStream.range(from, to)
				.mapToObj(definition::at)
				.collect(toList());
	}

	private int previousSection(int index) {
		while (index >= 0 && !isSection(index)) index--;
		return index;
	}

	private int nextSection(int index) {
		while (index < definition.size() && !isSection(index)) index++;
		return index;
	}

	private boolean isSection(int index) {
		return definition.at(index).type == FieldType.Section || definition.at(index).type == FieldType.Marker;
	}

}