package io.intino.monet.engine.edition.editors;

import io.intino.alexandria.logger.Logger;
import io.intino.monet.engine.edition.Edition;
import io.intino.monet.engine.edition.FormStore;
import io.intino.monet.engine.edition.Language;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.util.Collections.emptyList;

public class OptionEdition extends Edition {
	private final File thesaurusRepository;
	private final String optionsSource;
	protected List<String> options = emptyList();

	public OptionEdition(String name, FormStore store, Map<String, String> options, Language language) {
		super(name, store, options, language);
		this.thesaurusRepository = new File(options.get("thesaurus-repository"));
		this.optionsSource = options.get("values." + language);
	}

	@Override
	protected void onModified(String name) {
		if(!isDynamicSource()) return;
		if(Arrays.stream(getDependentFields()).anyMatch(f -> f.equals(name))) set("");
	}

	@Override
	public Edition init() {
		super.init();
		options = loadOptions();
		return this;
	}

	protected List<String> loadOptions() {
		if (!isDynamicSource()) return toList(optionsSource);
		String filename = sourceFilename();
		return filename != null ? optionsFromFile(filename) : emptyList();
	}

	protected boolean isDynamicSource() {
		return optionsSource.startsWith("@");
	}

	private List<String> optionsFromFile(String filename) {
		try {
			File thesaurus = new File(thesaurusRepository + "/" + filename);
			return thesaurus.exists() ? Files.readAllLines(thesaurus.toPath()) : emptyList();
		} catch (IOException e) {
			Logger.error(e);
			return emptyList();
		}
	}

	private String sourceFilename() {
		return Arrays.stream(getDependentFields()).map(this::storeValueOf).collect(Collectors.joining(""));
	}

	private String[] getDependentFields() {
		return optionsSource.substring(1).split(":");
	}

	public List<String> options() {
		return options;
	}

	public String get() {
		return (String) value();
	}

	public void set(String value) {
		this.store.put(name, value);
	}

	private List<String> toList(String values) {
		if (values == null) return emptyList();
		return Arrays.stream(values.split(";"))
				.map(String::trim)
				.collect(Collectors.toList());
	}

}
