package io.intino.monet.box.ui.displays.templates;

import io.intino.alexandria.Resource;
import io.intino.alexandria.core.Box;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.ui.displays.components.Text;
import io.intino.alexandria.ui.displays.components.TextCode;
import io.intino.monet.box.MonetBox;
import io.intino.monet.box.ui.displays.AttachmentRenderer;
import io.intino.monet.box.util.Formatters;
import io.intino.monet.box.util.OrderHelper;
import io.intino.monet.box.util.SearchHelper;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.edition.Field;
import io.intino.monet.engine.edition.FieldType;
import io.intino.monet.engine.edition.Form;
import io.intino.monet.engine.edition.editors.*;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class CheckListWizardStep extends AbstractCheckListWizardStep<MonetBox> {
    private Order order;
    private Form form;
    private Field field;
    private Consumer<Boolean> editingListener;
    private Consumer<Field> changeListener;
    private Consumer<Field> skipListener;
    private Consumer<Field> continueListener;

    public CheckListWizardStep(Box box) {
        super((MonetBox) box);
    }

    public CheckListWizardStep order(Order order) {
        this.order = order;
        return this;
    }

    public CheckListWizardStep form(Form form) {
        this.form = form;
        return this;
    }

    public CheckListWizardStep field(Field field) {
        this.field = field;
        return this;
    }

    public CheckListWizardStep onEditing(Consumer<Boolean> listener) {
        this.editingListener = listener;
        return this;
    }

    public CheckListWizardStep onChange(Consumer<Field> listener) {
        this.changeListener = listener;
        return this;
    }

    public CheckListWizardStep onSkip(Consumer<Field> listener) {
        this.skipListener = listener;
        return this;
    }

    public CheckListWizardStep onContinue(Consumer<Field> listener) {
        this.continueListener = listener;
        return this;
    }

    public Object value() {
        return field.edition().value();
    }

    @Override
    public void init() {
        super.init();
        textField.onChange(e -> updateText(e.value()));
        textField.onEnterPress(e -> updateTextAndContinue(e.value()));
        numberField.onChange(e -> updateNumber(e.value()));
        dateField.onChange(e -> updateDate(e.value()));
        optionField.onSelect(e -> updateOption(e.selection()));
        optionFieldSearch.onChange(e -> filterOptions(e.value()));
        optionFieldSearch.onEnterPress(e -> filterOptions(e.value()));
        multiOptionField.onSelect(e -> updateMultiOption(e.selection()));
        acceptValidation.onExecute(e -> acceptValidation());
        rejectValidation.onExecute(e -> rejectValidation());
        noteField.onChange(e -> updateNote(e.value()));
        notePhotoField.onUploading(e -> notifyEditing());
        notePhotoField.onChange(e -> updateNotePhoto(e.value()));
        photoField.onUploading(e -> notifyEditing());
        photoField.onChange(e -> updatePhoto(e.value()));
        photoLabelField.onChange(e -> updatePhotoLabel(e.value()));
        signatureField.onChange(e -> updateSignature(e.value()));
        skipStep.onExecute(e -> skip());
        optionalSkipSection.onExecute(e -> skip());
        optionalContinueSection.onExecute(e -> continueSection());
        continueSection.onExecute(e -> continueSection());
        packageFieldAdd.onExecute(e -> addPackage());
    }

    private void notifyEditing() {
        if (editingListener == null) return;
        editingListener.accept(true);
    }

    private void skip() {
        skipListener.accept(field);
    }

    private void addFileToPackage(int index, Object item) {
        try {
            if (item == null) {
                removeFileFromPackage(index);
                return;
            }
            PackageEdition.Package packageValue = field.edition().as(PackageEdition.class).get();
            Resource resource = (Resource) item;
            packageValue.add(resource.name(), resource.stream());
            refreshPackageField();
            notifyChange(field);
        } catch (IOException e) {
			Logger.error(e);
		}
	}

    private void removeFileFromPackage(int index) {
        PackageEdition.Package packageValue = field.edition().as(PackageEdition.class).get();
        List<File> files = packageValue.files();
        if (files.size() <= index) return;
        File file = files.get(index);
        file.delete();
        refreshPackageField();
        notifyChange(field);
    }

    private void updateText(String value) {
        if (!(field.edition() instanceof StringEdition)) return;
        StringEdition edition = field.edition().as(StringEdition.class);
        edition.set(format(value));
        notifyChange(field);
    }

    private void updateTextAndContinue(String value) {
        updateText(value);
        continueListener.accept(field);
    }

    private void updateNumber(Double value) {
        if (!(field.edition() instanceof NumberEdition)) return;
        NumberEdition edition = field.edition().as(NumberEdition.class);
        edition.set(value);
        notifyChange(field);
    }

    private void updateDate(Instant value) {
        if (!(field.edition() instanceof DateEdition)) return;
        DateEdition edition = field.edition().as(DateEdition.class);
        edition.set(LocalDate.ofInstant(value, ZoneId.of("UTC")));
        notifyChange(field);
    }

    private void updateOption(List<String> selection) {
        if (!(field.edition() instanceof OptionEdition)) return;
        OptionEdition edition = field.edition().as(OptionEdition.class);
        edition.set(selection.size() > 0 ? selection.get(0) : null);
        notifyChange(field);
        continueListener.accept(field);
    }

    private void filterOptions(String value) {
        if (!(field.edition() instanceof OptionEdition)) return;
        OptionEdition edition = field.edition().as(OptionEdition.class);
        optionField.clear();
        optionField.addAll(SearchHelper.filter(edition.options(), value));
    }

    private void updateMultiOption(List<String> selection) {
        if (!(field.edition() instanceof OptionMultipleEdition)) return;
        OptionMultipleEdition edition = field.edition().as(OptionMultipleEdition.class);
        List<String> options = edition.options();
        edition.set(selection.stream().filter(options::contains).toArray(String[]::new));
        notifyChange(field);
    }

    private void acceptValidation() {
        if (!(field.edition() instanceof ValidationEdition)) return;
        ValidationEdition edition = field.edition().as(ValidationEdition.class);
        edition.set(edition.options().get(0));
        notifyChange(field);
        continueListener.accept(field);
    }

    private void rejectValidation() {
        if (!(field.edition() instanceof ValidationEdition)) return;
        ValidationEdition edition = field.edition().as(ValidationEdition.class);
        edition.set(edition.options().get(1));
        notifyChange(field);
        continueListener.accept(field);
    }

    private void updatePhoto(Object value) {
        try {
            if (!(field.edition() instanceof ImageEdition)) return;
            URL image = urlOf(value);
            ImageEdition edition = field.edition().as(ImageEdition.class);
            edition.set(image != null ? Paths.get(image.toURI()).toFile() : null, photoLabelField.value());
            photoField.value(edition.get() != null ? edition.get().file : null);
            notifyChange(field);
        } catch (URISyntaxException e) {
            Logger.error(e);
        }
    }

    private void updatePhotoLabel(Object value) {
        if (!(field.edition() instanceof ImageEdition)) return;
        ImageEdition edition = field.edition().as(ImageEdition.class);
        ImageEdition.Image image = edition.get();
        edition.set(image != null ? image.file : null, format(photoLabelField.value()));
        notifyChange(field);
    }

    private void updateSignature(Object value) {
        try {
            if (!(field.edition() instanceof SignatureEdition)) return;
            URL signature = urlOf(value);
            SignatureEdition edition = field.edition().as(SignatureEdition.class);
            edition.set(signature != null ? Paths.get(signature.toURI()).toFile() : null);
            signatureField.value(edition.get() != null ? edition.get().file : null);
            notifyChange(field);
        } catch (URISyntaxException e) {
            Logger.error(e);
        }
    }

    private void updateNote(Object value) {
        if (!(field.edition() instanceof NoteEdition)) return;
        NoteEdition edition = field.edition().as(NoteEdition.class);
        edition.set(format(noteField.value()));
        notifyChange(field);
    }

    private void updateNotePhoto(Object value) {
        try {
            if (!(field.edition() instanceof NoteEdition)) return;
            URL image = urlOf(value);
            NoteEdition edition = field.edition().as(NoteEdition.class);
            edition.set(noteField.value(), image != null ? Paths.get(image.toURI()).toFile() : null);
            notePhotoField.value(edition.get() != null ? edition.get().file : null);
            notifyChange(field);
        } catch (URISyntaxException e) {
            Logger.error(e);
        }
    }

    private void continueSection() {
        if (!(field.edition() instanceof SectionEdition)) return;
        SectionEdition edition = field.edition().as(SectionEdition.class);
        edition.set();
        continueListener.accept(field);
    }

    private URL urlOf(Object value) {
        if (field.type() == FieldType.Image && value == null) return null;
        if (field.type() == FieldType.Signature && value == null) return null;
        if (!(value instanceof Resource)) return null;
        Resource resource = (Resource) value;
        return urlOf(value, username() + "_" + field.name().replace(" ", "") + extensionOf(resource.name()));
    }

    private URL urlOf(Object value, String name) {
        try {
            if (field.type() == FieldType.Image && value == null) return null;
            if (field.type() == FieldType.Signature && value == null) return null;
            if (!(value instanceof Resource)) return null;
            Resource resource = (Resource) value;
            File image = new File(System.getProperty("java.io.tmpdir") + "/" + name);
            Files.write(image.toPath(), resource.bytes(), StandardOpenOption.CREATE);
            try { resource.stream().reset(); } catch (IOException ignored) {}
            return image.toURI().toURL();
        } catch (IOException e) {
            Logger.error(e);
            return null;
        }
    }

    private String extensionOf(String name) {
        if (!name.contains(".")) return name;
        return name.substring(name.lastIndexOf("."));
    }

    private void notifyChange(Object value) {
        changeListener.accept(field);
    }

    @Override
    public void refresh() {
        super.refresh();
        skipStep.visible(field.edition().isOptional());
        refreshHeader(title, info);
        refreshAttachmentDialog();
        refreshEdition();
    }

    private void refreshAttachmentDialog() {
        File attachment = field.attachment();
        boolean existAttachment = attachment != null && attachment.exists();
        attachmentStamp.visible(existAttachment);
        if (!existAttachment) return;
        AttachmentRenderer display = new AttachmentRenderer(box());
        attachmentStamp.display(display);
        display.order(order);
        display.form(form);
        display.field(field);
        display.attachment(attachment);
        display.refresh();
    }

    private void refreshEdition() {
        refreshSectionField();
        refreshOtherFields();
    }

    private void refreshSectionField() {
        sectionStepFields.visible(field.type() == FieldType.Section || field.type() == FieldType.Marker);
        if (!sectionStepFields.isVisible()) return;
        boolean optional = field.edition().isOptional();
        refreshHeader(sectionTitle, sectionInfo);
        optionalSectionBlock.visible(optional);
        defaultSectionBlock.visible(!optional);
    }

    private void refreshOtherFields() {
        otherStepFields.visible(field.type() != FieldType.Section && field.type() != FieldType.Marker);
        if (!otherStepFields.isVisible()) return;
        refreshPhotoField();
        refreshSignatureField();
        refreshNoteField();
        refreshRemarkField();
        refreshPackageField();
        refreshDateField();
        refreshTextField();
        refreshOptionField();
        refreshMultiOptionField();
        refreshValidationField();
        refreshNumberField();
    }

    private void refreshPhotoField() {
        try {
            photoFieldBlock.visible(field.type() == FieldType.Image);
            if (!photoFieldBlock.isVisible()) return;
            ImageEdition imageEdition = field.edition().as(ImageEdition.class);
            ImageEdition.Image photo = imageEdition.get();
            photoField.value(photo != null && photo.file != null ? photo.file.toURI().toURL() : null);
            photoLabelField.value(!imageEdition.isEmpty() && photo != null ? photo.label.trim() : null);
        } catch (MalformedURLException e) {
            Logger.error(e);
        }
    }

    private void refreshSignatureField() {
        try {
            signatureFieldBlock.visible(field.type() == FieldType.Signature);
            if (!signatureFieldBlock.isVisible()) return;
            SignatureEdition signatureEdition = field.edition().as(SignatureEdition.class);
            SignatureEdition.Signature signature = signatureEdition.get();
            signatureField.value(signature != null && signature.file != null ? signature.file.toURI().toURL() : null);
        } catch (MalformedURLException e) {
            Logger.error(e);
        }
    }

    private void refreshNoteField() {
        try {
            noteFieldBlock.visible(field.type() == FieldType.Note);
            if (!noteFieldBlock.isVisible()) return;
            NoteEdition edition = field.edition().as(NoteEdition.class);
            NoteEdition.Note note = edition.get();
            noteField.value(!edition.isEmpty() && note != null ? note.label.replace("<br/>", "\n") : null);
            notePhotoField.value(note != null && note.file != null ? note.file.toURI().toURL() : null);
        } catch (MalformedURLException e) {
            Logger.error(e);
        }
    }

    private void refreshRemarkField() {
    }

    private void refreshPackageField() {
        packageFieldBlock.visible(field.type() == FieldType.Package);
        if (!packageFieldBlock.isVisible()) return;
        packageField.clear();
        PackageEdition edition = field.edition().as(PackageEdition.class);
        edition.set();
        PackageEdition.Package packageData = edition.get();
        List<File> files = packageData.files();
        for (int i = 0; i< files.size(); i++) {
            fill(files.get(i), i, packageField.add());
        }
    }

    private void fill(File file, int index, CheckListWizardStepPackageFile display) {
        try {
            display.value(file != null ? file.toURI().toURL() : null);
            display.onChange(value -> addFileToPackage(index, value));
            display.onUploading(value -> notifyEditing());
            display.refresh();
        } catch (MalformedURLException e) {
            Logger.error(e);
        }
    }

    private void addPackage() {
        if (!(field.edition() instanceof PackageEdition)) return;
        fill(null, field.edition().as(PackageEdition.class).get().files().size(), packageField.add());
    }

    private void refreshDateField() {
        dateField.visible(field.type() == FieldType.Date);
        if (!dateField.isVisible()) return;
        DateEdition edition = field.edition().as(DateEdition.class);
        dateField.value(!edition.isEmpty() ? Formatters.instantOf(edition.get()) : null);
        if (edition.mode() == DateEdition.Mode.Future) dateField.range(Instant.now(), null);
        else if (edition.mode() == DateEdition.Mode.Past) dateField.range(null, Instant.now());
    }

    private void refreshTextField() {
        textField.visible(field.type() == FieldType.String);
        if (!textField.isVisible()) return;
        StringEdition edition = field.edition().as(StringEdition.class);
        textField.value(!edition.isEmpty() ? edition.get() : null);
    }

    private void refreshOptionField() {
        optionFieldBlock.visible(false);
        if (field.type() != FieldType.Option) return;
        OptionEdition edition = field.edition().as(OptionEdition.class);
        List<String> options = edition.options();
        optionFieldSearch.value(null);
        optionFieldSearch.visible(options.size() > 5);
        optionField.clear();
        optionField.addAll(options);
        String selected = !edition.isEmpty() ? edition.get() : null;
        if (selected != null) optionField.selection(selected);
        else optionField.selection(new ArrayList<>());
        selectedOptionField.value(selected);
        optionFieldBlock.visible(true);
    }

    private void refreshMultiOptionField() {
        multiOptionField.visible(field.type() == FieldType.MultiOption);
        if (!multiOptionField.visible()) return;
        OptionMultipleEdition edition = field.edition().as(OptionMultipleEdition.class);
        List<String> options = edition.options();
        multiOptionField.clear();
        multiOptionField.addAll(options);
        multiOptionField.readonly(options.size() <= 0);
        List<String> selection = Arrays.asList(edition.get());
        multiOptionField.selection(selection.stream().filter(options::contains).collect(Collectors.toList()));
    }

    private void refreshValidationField() {
        validationFieldBlock.visible(false);
        if (field.type() != FieldType.Validation) return;
        ValidationEdition edition = field.edition().as(ValidationEdition.class);
        List<String> options = edition.options();
        acceptValidation.title(options.get(0));
        rejectValidation.title(options.get(1));
        validationFieldBlock.visible(true);
    }

    private void refreshNumberField() {
        numberField.visible(field.type() == FieldType.Number);
        if (!numberField.isVisible()) return;
        NumberEdition edition = field.edition().as(NumberEdition.class);
        numberField.value(!edition.isEmpty() ? edition.get() : null);
        numberField.range(edition.min(), edition.max());
    }

    private void refreshHeader(Text<?, ?> title, TextCode<?, ?> info) {
        String description = OrderHelper.replaceInputs(order, field.description());
        String header = OrderHelper.replaceInputs(order, field.header());
        title.value((header != null && !header.isEmpty() ? header + ": " : "") + field.title());
        info.visible(!description.isEmpty());
        if (info.isVisible()) info.value(description);
    }

    private String format(String value) {
        if (value == null) return value;
        return value.replace("\n", "<br/>").replace("\t", "").trim();
    }

}