package io.intino.monet.box;

import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.ui.services.AuthService;
import io.intino.alexandria.ui.services.auth.Space;
import io.intino.amidas.accessor.alexandria.core.AmidasOauthAccessor;
import io.intino.monet.archetype.Archetype;
import io.intino.monet.box.commands.Commands;
import io.intino.monet.box.commands.CommandsFactory;
import io.intino.monet.box.workreports.WorkReportInfoProvider;
import io.intino.monet.engine.OrderTypes;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;

public class MonetBox extends AbstractBox {

	private final Archetype archetype;
	private CommandsFactory commandsFactory;
	private WorkReportGenerator workReportGenerator;
	private WorkReportInfoProvider workReportInfoProvider;
	private OrderApi orderApi;

	public MonetBox(String[] args) {
		this(new MonetConfiguration(args));
	}

	public MonetBox(MonetConfiguration configuration) {
		super(configuration);
		this.archetype = new Archetype(configuration.home());
	}

	@Override
	public io.intino.alexandria.core.Box put(Object o) {
		super.put(o);
		if (o instanceof WorkReportGenerator) this.workReportGenerator = (WorkReportGenerator) o;
		if (o instanceof WorkReportInfoProvider) this.workReportInfoProvider = (WorkReportInfoProvider) o;
		return this;
	}

	public Archetype archetype() {
		return archetype;
	}

	public void beforeStart() {
		commandsFactory = new CommandsFactory(this);
		orderApi = new OrderApi(archetype);
		if (workReportInfoProvider == null) workReportInfoProvider = new WorkReportInfoProvider();
		OrderTypes.init(archetype.definitions().getOrderTypes());
	}

	public void afterStart() {
	}

	public void beforeStop() {
	}

	public void afterStop() {
		orderApi.stop();
	}

	public void commitDatabases() {
		orderApi.commit();
	}

	public OrderApi orderApi() {
		return orderApi;
	}

	public void onOrderFinished(OrderFinishedListener listener) {
		orderApi.onOrderFinished(listener);
	}

	public WorkReportGenerator workReportGenerator() {
		return workReportGenerator;
	}

	public WorkReportInfoProvider workReportInfoProvider() {
		return workReportInfoProvider;
	}

	public <F extends Commands> F commands(Class<F> clazz) {
		return commandsFactory.command(clazz);
	}

	@Override
	protected AuthService authService(URL authServiceUrl) {
		try {
			if (authServiceUrl == null) return null;
			return new AmidasOauthAccessor(new Space(URI.create(configuration().url()).toURL()), authServiceUrl);
		} catch (MalformedURLException e) {
			Logger.error(e);
			return null;
		}
	}

	public URL logoUrl() {
		try {
			String logo = configuration().logo();
			File logoFile;
			if (logo != null && !logo.isEmpty()) logoFile = new File(logo);
			else logoFile = archetype().configuration().getLogo();
			return logoFile.exists() ? logoFile.toURI().toURL() : MonetBox.class.getResource("/logo.png");
		}
		catch (MalformedURLException e) {
			Logger.error(e);
			return null;
		}
	}

}