package io.intino.monet.box.workreports;

import io.intino.alexandria.Resource;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.office.components.Image;
import io.intino.alexandria.office.components.ImageView;
import io.intino.monet.box.ImageViewFactory;
import io.intino.monet.box.WorkReportGenerator;
import io.intino.monet.box.util.Resources;
import io.intino.monet.box.util.WorkReportHelper;
import io.intino.monet.engine.CheckListProvider;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.OrderTypes;
import io.intino.monet.engine.OrderTypes.Checklist;
import io.intino.monet.engine.OrderTypes.Checklist.Annex;
import io.intino.monet.engine.WorkReport;
import io.intino.monet.engine.edition.FormStore;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class WorkReportBuilder {

    private static final String MAIN_TEMPLATE_NAME = "000000_main";

    private final WorkReportGenerator.Archetype archetype;
    private WorkReport workReport;
    private WorkReportInfoProvider infoProvider;

    private Checklist checklist;
    private final Map<String, WorkReportTemplate> templates = new TreeMap<>();

    private ImageViewFactory imageViewFactory = new ImageViewFactory.Default();
    private boolean computeCalculations = true;

    public WorkReportBuilder(WorkReportGenerator.Archetype archetype) {
        this.archetype = archetype;
    }

    public Map<String, WorkReportTemplate> build() throws WorkReportException {
        validate();
        templates.clear();
        buildDocuments();
        return templates;
    }

    private void buildDocuments() throws WorkReportException {
        createMainTemplate();
        fillWorkReportTemplates();
    }

    private void fillWorkReportTemplates() throws WorkReportException {
        setInputs(mainTemplate());
        setBasicInfo(mainTemplate());

        Map<String, String> workReportChecks = new TreeMap<>(computeCalculations ? withCalculations() : workReport.attributes());
        setChecksAndCalculations(workReportChecks);
        setImages();

        handleNotDefinedImages();
        setDefaultValuesForNotProcessedChecks(workReportChecks);
    }

    protected void setChecksAndCalculations(Map<String, String> workReportChecks) throws WorkReportException {

        for(Map.Entry<String, String> checkEntry : workReportChecks.entrySet()) {

            String name = checkEntry.getKey();
            String value = checkEntry.getValue();

            if(value == null) continue;

            Checklist.Field check = getCheck(name);

            WorkReportTemplate template = check != null ? templateOf(check) : templateOf(name);

            if(template == null) {
                Logger.warn("Template of " + name + " not found...");
                return;
            }

            setTemplateFields(name, value, template);
        }
    }

    private void setTemplateFields(String name, String value, WorkReportTemplate template) {
        value = value.replace("<br/>", "\n");
        template.set(checkName(name), value);
    }

    private void setAnnexFields(Checklist.Field field, Annex annex, String value, WorkReportTemplate template) {
        if(annex == null || annex.type() == Annex.Type.shared) return;
        Checklist.AnnexInstance annexInstance = (Checklist.AnnexInstance) annex;
        for(Map.Entry<String, String> attribute : annexInstance.attributes(workReport().language()).entrySet()) {
            template.set(attribute.getKey(), attribute.getValue());
        }

        template.set("$value", isValidValue(value) ? value : "");
        template.set("$instance.index", template.extraInfo().get("$instance.index"));

        if(field.marker() != null) setMarkerParameters(field, template);
    }

    private static final Set<String> InvalidValues = Set.of(String.valueOf(FormStore.skipped), String.valueOf(FormStore.disabled), String.valueOf(FormStore.hidden));
    private boolean isValidValue(String value) {
        return !InvalidValues.contains(value);
    }

    private void setMarkerParameters(Checklist.Field field, WorkReportTemplate template) {
        for(Map.Entry<String, String> entry : field.marker().entries().entrySet()) {
            String name = entry.getKey();
            if(!matchesLanguageOrIsGlobal(name, workReport.language())) continue;
            name = name.endsWith("." + workReport.language()) ? name.replace("." + workReport.language(), "") : name;
            template.set("$marker." + name, entry.getValue());
        }
    }

    private boolean matchesLanguageOrIsGlobal(String name, String language) {
        int langStart = name.lastIndexOf('.');
        if(langStart < 0) return true; // No language -> Global
        String attribLang = name.substring(langStart + 1);
        if(Locale.forLanguageTag(attribLang) == null) return true; // No language -> Global
        return language.equals(attribLang);
    }

    private void setDefaultValuesForNotProcessedChecks(Map<String, String> workReportChecks) throws WorkReportException {
        Set<String> checks = checklist.fields().stream()
                .map(f -> f.name)
                .collect(Collectors.toSet());

        Set<String> notProcessedChecks = new HashSet<>(checks);
        notProcessedChecks.removeAll(workReportChecks.keySet());

        for(WorkReportTemplate template : templates.values()) {
            for(String check : notProcessedChecks) {
                if(!isCheckOfTemplate(template, check)) continue;
                if(check.contains("+")) check = check.substring(0, check.indexOf('+'));
                if(!template.contains(check)) template.set(check, "");
            }
        }

        Set<String> processedChecks = new HashSet<>(checks);
        processedChecks.retainAll(workReportChecks.keySet());

        for(String check : processedChecks) {
            Checklist.Field field = getCheck(check);
            if(field == null) continue;
            WorkReportTemplate template = templateOf(field);
            if(template.empty()) continue;
            setAnnexFields(field, field.annex(), workReportChecks.get(check), template);
        }
    }

    private Checklist.Field getCheck(String name) {
        return checklist.fields().stream().filter(f -> f.name.equals(name)).findFirst().orElse(null);
    }

    protected void setBasicInfo(WorkReportTemplate mainTemplate) {
        infoProvider.info().forEach(mainTemplate::set);
    }

    protected void setInputs(WorkReportTemplate mainTemplate) {
        Order order = workReport.order();
        if (order.inputMap() != null) order.inputMap().forEach(mainTemplate::set);
    }

    private Map<String, String> withCalculations() {
        return WorkReportHelper.addCalculationsTo(workReport.attributes(), workReport.order(), archetype.orderTypes());
    }

    protected void setImages() {
        for (Resource image : workReport.images()) setImage(image);
    }

    private void handleNotDefinedImages() {
        for(WorkReportTemplate template : templates.values()) {
            imagesOfTemplate(template).forEach(check -> tryPut(check, template));
        }
    }

    private void tryPut(Checklist.Field check, WorkReportTemplate template) {
        String name = check.name.contains("+") ? check.name.substring(0, check.name.indexOf('+')) : check.name;
        if(template.containsImage(name)) return;
        byte[] imageData = getImageForTemplateState(template);
        if(imageData.length == 0) return;
        try {
            setImage(name, check.annex(), template, imageData);
        } catch (IOException e) {
            Logger.error(e);
        }
    }

    private Stream<Checklist.Field> imagesOfTemplate(WorkReportTemplate template) {
        return checklist.fields().stream().filter(this::isImageCheck).filter(c -> isCheckOfTemplate(template, c.name));
    }

    private static boolean isCheckOfTemplate(WorkReportTemplate template, String checkName) {
        if(checkName.contains("+")) return checkName.endsWith("+" + template.name());
        return template.isMainTemplate();
    }

    private boolean isImageCheck(Checklist.Field field) {
        return field.type == Checklist.Type.Image || field.type == Checklist.Type.Note || field.type == Checklist.Type.Signature;
    }

    protected void setImage(Resource image) {
        try {
            Checklist.Field check = getCheck(image.name());
            Annex annex = check == null ? null : check.annex();

            byte[] imageData = image.bytes();

            if(annex != null && annex.type() == Annex.Type.instance && imageData.length == 0) return;

            WorkReportTemplate template = templateOf(check);
            if(template == null) {
                Logger.warn("Template of " + image.name() + " not found...");
                return;
            }

            if(imageData.length == 0) imageData = getImageForTemplateState(template);

            if(imageData.length == 0) return;

            setImage(image.name(), check == null ? null : check.annex(), template, imageData);

        } catch (Throwable e) {
            Logger.error("Failed to create image " + image.name() + ": " + e.getMessage(), e);
        }
    }

    private void setImage(String name, Annex annex, WorkReportTemplate template, byte[] imageData) throws IOException {
        ImageView imageView = imageViewFactory.create(new Image(imageData));
        template.set(checkName(name), imageView);
        if(annex != null && annex.type() == Annex.Type.instance)
            template.set("$value", imageView);
    }

    private byte[] getImageForTemplateState(WorkReportTemplate template) {
        // Image is not set + the template is the main one => blank image
        // Image is not set + the template is NOT empty => blank image
        // Image is not set + the template is an annex + template is empty => skip
        return !template.empty() || template.isMainTemplate() ? blankImage() : new byte[0];
    }

    private byte[] blankImage() {
        try(InputStream inputStream = Resources.emptyImage()) {
            return inputStream.readAllBytes();
        } catch (IOException e) {
            Logger.error("Could not read empty image: " + e.getMessage(), e);
            return new byte[0];
        }
    }

    private void createMainTemplate() {
        WorkReportTemplate template = new WorkReportTemplate(
                MAIN_TEMPLATE_NAME,
                WorkReportHelper.template(archetype.orderTypes(), workReport.area(), workReport.order(), workReport.language())
        );

        template.setAsMainTemplate();

        templates.put(MAIN_TEMPLATE_NAME, template);
    }

    private WorkReportTemplate templateOf(Checklist.Field check) throws WorkReportException {
        return check == null ? mainTemplate() : templateOf(check.annex());
    }

    private WorkReportTemplate templateOf(Annex annex) throws WorkReportException {
        if(annex == null) return mainTemplate();
        return annex.type() == Annex.Type.instance ? newWorkReportTemplate(annex) : sharedWorkReportTemplate(annex);
    }

    private WorkReportTemplate sharedWorkReportTemplate(Annex annex) throws WorkReportException {
        if(!templates.containsKey(annex.name))
            templates.put(annex.name, templateOfFilename(annex.name, orderType().annexFilenameOf(annex.name)));
        return templates.get(annex.name);
    }

    private WorkReportTemplate newWorkReportTemplate(Annex annex) throws WorkReportException {
        Checklist.AnnexInstance annexInstance = (Checklist.AnnexInstance) annex;
        if(!templates.containsKey(annexInstance.instanceName)) {
            WorkReportTemplate template = templateOfFilename(annexInstance.instanceName, orderType().annexFilenameOf(annex.name));
            templates.put(annexInstance.instanceName, template);
            template.extraInfo().put("$instance.index", String.valueOf(countTemplateInstances(annexInstance.name)));
        }
        return templates.get(annexInstance.instanceName);
    }

    private int countTemplateInstances(String name) {
        return (int) templates.keySet().stream().filter(k -> k.startsWith(name)).count();
    }

    private WorkReportTemplate templateOf(String name) {
        if(!name.contains("+")) return mainTemplate();
        String templateId = name.substring(name.indexOf('+') + 1);
        return templates.get(templateId);
    }

    private WorkReportTemplate templateOfFilename(String name, String filename) {
        return new WorkReportTemplate(
                name,
                new File(mainTemplate().file().getParentFile(), filename));
    }

    protected WorkReportTemplate mainTemplate() {
        return templates.get(MAIN_TEMPLATE_NAME);
    }

    protected String checkName(String name) {
        return !name.contains("+") ? name : name.substring(0, name.indexOf('+'));
    }

    protected void validate() throws WorkReportException {
        if(workReport == null) throw new WorkReportException("WorkReport is null");
        if(workReport.order() == null) throw new WorkReportException("WorkReport orderId is null");
        if(workReport.language() == null) throw new WorkReportException("Language is not specified");
        if((checklist = new CheckListProvider().findCheckList(workReport.order().code())) == null) throw new WorkReportException("No checklist found for order " + workReport.order().code());
    }

    @SuppressWarnings("unchecked")
    public <T extends WorkReport> T workReport() {
        return (T) workReport;
    }

    private OrderTypes.Record orderType() throws WorkReportException {
        Order order = workReport.order();
        OrderTypes.Record orderType = OrderTypes.of(order.code());
        if(orderType == null) throw new WorkReportException("OrderType not found for order " + order.code());
        return orderType;
    }

    public WorkReportBuilder workReport(WorkReport workReport) {
        this.workReport = workReport;
        return this;
    }

    public WorkReportBuilder infoProvider(WorkReportInfoProvider infoProvider) {
        this.infoProvider = infoProvider;
        return this;
    }

    public WorkReportBuilder imageViewFactory(ImageViewFactory imageViewFactory) {
        this.imageViewFactory = imageViewFactory == null ? new ImageViewFactory.Default() : imageViewFactory;
        return this;
    }

    public WorkReportBuilder computeCalculations(boolean computeCalculations) {
        this.computeCalculations = computeCalculations;
        return this;
    }

    public static class WorkReportException extends Exception {
        public WorkReportException(String message) {
            super(message);
        }
    }

}
