package io.intino.monet.box.ui.displays.templates;

import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.ui.displays.UserMessage;
import io.intino.monet.box.MonetBox;
import io.intino.monet.box.WorkReportGenerator;
import io.intino.monet.box.commands.order.OrderCommands;
import io.intino.monet.box.ui.displays.ProgressDisplay;
import io.intino.monet.box.ui.displays.notifiers.OrderTemplateNotifier;
import io.intino.monet.box.util.FormHelper;
import io.intino.monet.box.util.OrderHelper;
import io.intino.monet.box.util.WorkReportCalculator;
import io.intino.monet.box.util.WorkReportHelper;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.OrderTypes;
import io.intino.monet.engine.edition.*;
import io.intino.monet.engine.edition.editors.DateEdition;
import io.intino.monet.engine.edition.editors.NumberEdition;
import io.intino.monet.engine.edition.editors.OptionMultipleEdition;
import io.intino.monet.engine.edition.editors.SectionEdition;
import org.apache.commons.io.FileUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.LocalDate;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.BiConsumer;
import java.util.stream.Stream;

public class OrderTemplate extends AbstractOrderTemplate<OrderTemplateNotifier, MonetBox> {
	private Order order;
	private String author;
	private Form form;
	private WorkReportCalculator formulaCalculator;
	private BiConsumer<File, Form> completeListener;
	private boolean testEnvironment = false;
	private Field previous;
	private Field current;

	public void orderId(String orderId) {
		order(box().orderApi().order(orderId));
	}

	public void author(String author) {
		this.author = author;
	}

	public OrderTemplate(MonetBox box) {
		super(box);
	}

	public void open(String id) {
		order(box().orderApi().order(id));
		refresh();
	}

	public OrderTemplate order(Order order) {
		this.order = order;
		form = order != null ? new Form(formDefinition(), store()) : null;
		current = form != null ? form.start(Language.valueOf(language())) : null;
		createFormulaCalculator();
		return this;
	}

	public void testEnvironment(boolean value) {
		this.testEnvironment = value;
	}

	public void onComplete(BiConsumer<File, Form> listener) {
		this.completeListener = listener;
	}

	@Override
	public void init() {
		super.init();
		editionBlock.onInit(e -> initEdition());
		editionBlock.onShow(e -> refreshEdition());
		previewBlock.onShow(e -> refreshPreview());
		previewBlock.onHide(e -> removePreview());
		indexBlock.onShow(e -> refreshIndex());
		current = form != null ? form.start(Language.valueOf(language())) : null;
		initToolbar();
		refresh();
	}

	@Override
	public void refresh() {
		super.refresh();
		orderNotFoundBlock.visible(order == null);
		orderFoundBlock.visible(order != null);
		orderCompletedBlock.visible(false);
		if (order == null) return;
		refreshResolveOrderDialog();
		if (editionBlock.isVisible()) refreshEdition();
		else if (previewBlock.isVisible()) refreshPreview();
		else if (indexBlock.isVisible()) refreshIndex();
	}

	private void initToolbar() {
		previousStep.onExecute(e -> {
			previousStep();
			if (!editionBlock.isVisible()) viewSelector.select("editionOption");
		});
		nextStep.onExecute(e -> continueStep(current));
		restartOrder.onExecute(e -> reset());
		terminateOrder.onExecute(e -> notifyComplete());
	}

	private void continueStep(Field field) {
		Edition edition = field.edition();
		setValueIfEmpty(field);
		if (edition.isOptional() && (edition.isEmpty() || edition.isNull())) skip(field);
		else nextStep();
		if (!editionBlock.isVisible()) viewSelector.select("editionOption");
	}

	private void setValueIfEmpty(Field field) {
		setNumberValueIfEmpty(field);
		setMultiOptionValueIfEmpty(field);
		setDateValueIfEmpty(field);
		setSectionValueIfEmpty(field);
	}

	private void setNumberValueIfEmpty(Field field) {
		if (field.type() != FieldType.Number) return;
		NumberEdition edition = field.edition().as(NumberEdition.class);
		if (edition.isEmpty() || edition.isNull()) edition.set(edition.min());
	}

	private void setMultiOptionValueIfEmpty(Field field) {
		if (field.type() != FieldType.MultiOption) return;
		OptionMultipleEdition edition = field.edition().as(OptionMultipleEdition.class);
		if (edition.isEmpty() || edition.isNull()) edition.set();
	}

	private void setDateValueIfEmpty(Field field) {
		if (field.type() != FieldType.Date) return;
		DateEdition edition = field.edition().as(DateEdition.class);
		if (edition.isEmpty() || edition.isNull()) edition.set(LocalDate.now());
	}

	private void setSectionValueIfEmpty(Field field) {
		if (field.type() != FieldType.Section && field.type() != FieldType.Marker) return;
		SectionEdition edition = field.edition().as(SectionEdition.class);
		if (edition.isEmpty() || edition.isNull()) edition.set();
	}

	private void previousStep() {
		if (current == null) current = previous;
		else if (current.hasPrev()) current = current.prev();
		renderWizardStep();
	}

	private void nextStep() {
		previous = current;
		current = current.hasNext() ? current.next() : null;
		applyFormula(current);
		renderWizardStep();
	}

	private void renderWizardStep() {
		editionBlock.finalBlock.visible(current == null);
		editionBlock.stepBlock.visible(current != null);
		refreshStepDisplay(current);
		refreshState(current);
	}

	private void refreshStepDisplay(Field field) {
		if (field == null) return;
		editionBlock.stepBlock.stepStamp.order(order);
		editionBlock.stepBlock.stepStamp.form(form);
		editionBlock.stepBlock.stepStamp.field(field);
		editionBlock.stepBlock.stepStamp.refresh();
	}

	private void skip(Field field) {
		field.edition().skip();
		nextStep();
	}

	private void refreshStateAndSave(Field field) {
		refreshState(field);
		save();
	}

	private void refreshState(Field field) {
		refreshProgress();
		restartOrder.readonly(form.progress() <= 0);
		previousStep.readonly(field != null && !field.hasPrev());
		nextStep.readonly(field == null || !isFilled(field));
		terminateOrder.readonly(!form.isCompleted());
	}

	private boolean isFilled(Field field) {
		Edition edition = field.edition();
		if (edition.isOptional()) return true;
		if (field.type() == FieldType.Date) return true;
		if (field.type() == FieldType.Number) return true;
		if (field.type() == FieldType.MultiOption) return true;
		return !edition.isEmpty() && !edition.isNull();
	}

	private void initEdition() {
		editionBlock.stepBlock.stepStamp.onChange(this::refreshStateAndSave);
		editionBlock.stepBlock.stepStamp.onSkip(this::skip);
		editionBlock.stepBlock.stepStamp.onContinue(this::continueStep);
		editionBlock.finalBlock.viewIndex.onExecute(e -> viewSelector.select("indexOption"));
		editionBlock.finalBlock.viewWorkReport.onExecute(e -> viewSelector.select("previewOption"));
	}

	private void refreshEdition() {
		OrderTypes.Record record = OrderTypes.of(order.code());
		editionBlock.orderInfo.value(record != null ? OrderHelper.replaceInputs(order, record.hint(language())) : null);
		editionBlock.orderInfo.visible(editionBlock.orderInfo.value() != null && !editionBlock.orderInfo.value().isEmpty());
		renderWizardStep();
		removePreview();
	}

	private void refreshPreview() {
		removePreview();
		previewBlock.workReportPreview.value(previewReport());
		refreshState(current);
	}

	@SuppressWarnings("ResultOfMethodCallIgnored")
	private void removePreview() {
		try {
			if (previewBlock.workReportPreview == null) return;
			URL value = previewBlock.workReportPreview.value();
			if (value == null) return;
			File file = new File(value.toURI());
			if (!file.exists()) return;
			file.delete();
		} catch (URISyntaxException e) {
			Logger.error(e);
		}
	}

	private void refreshIndex() {
		generatingMessage.visible(true);
		indexBlock.indexEntriesBlock.hide();
		indexBlock.indexEntriesBlock.indexEntries.clear();
		form.editions().stream().filter(e -> !e.isHidden() && !e.isDisabled()).forEach(e -> fill(e, indexBlock.indexEntriesBlock.indexEntries.add()));
		generatingMessage.visible(false);
		indexBlock.indexEntriesBlock.show();
		refreshState(current);
		removePreview();
	}

	private void fill(Edition edition, CheckListWizardIndexEntry display) {
		display.form(form);
		display.edition(edition);
		display.onSelect(e -> selectStep(e.name()));
		display.refresh();
	}

	private void selectStep(String name) {
		current = form.field(name);
		viewSelector.select("editionOption");
	}

	private void notifyComplete() {
		save();
		terminateOrder.disable();
		File report = resolveOrder();
		if (report == null) {
			terminateOrder.enable();
			return;
		}
		if (completeListener != null) completeListener.accept(report, form);
		notifier.finish();
		removePreview();
		removeStore();
		showOrderCompleted();
	}

	private void removeStore() {
		if (order.isStoreDefined()) return;
		reset();
	}

	private File resolveOrder() {
		generatingMessage.visible(true);
		OrderCommands commands = box().commands(OrderCommands.class);
		if (!commands.canResolve(order, form, author())) {
			notifyUser(translate("Could not resolve order. Check parameters."), UserMessage.Type.Error);
			return null;
		}
		File result = commands.resolve(order, form, testEnvironment, language(), author());
		if (result == null) notifyUser(translate("Could not resolve order"), UserMessage.Type.Error);
		generatingMessage.visible(false);
		return result;
	}

	private void showOrderCompleted() {
		orderFoundBlock.visible(false);
		orderCompletedBlock.visible(true);
	}

	private File previewReport() {
		generatingMessage.visible(true);
		OrderCommands commands = box().commands(OrderCommands.class);
		if (!commands.canResolve(order, form, author())) {
			notifyUser(translate("Could not preview order. Check parameters."), UserMessage.Type.Error);
			return null;
		}
		File result = commands.preview(order, form, language(), author());
		if (result == null) notifyUser(translate("Could not resolve order"), UserMessage.Type.Error);
		generatingMessage.visible(false);
		return result;
	}

	private String author() {
		return author != null ? author : username();
	}

	private FormDefinition formDefinition() {
		return FormHelper.formDefinition(box(), order);
	}

	private FormStore store() {
		File directory = box().archetype().repository().workorders().getStoreDirectory(order.store());
		LocalFormStore result = new LocalFormStore(directory);
		try {
			result.load();
		} catch (IOException ignored) {
		}
		fillInputs(result);
		return result;
	}

	private void fillInputs(LocalFormStore result) {
		order.inputMap().forEach(result::put);
	}

	private void refreshResolveOrderDialog() {
		progress.clear();
		viewSelector.select("editionOption");
	}

	private void save() {
		try {
			((LocalFormStore) form.store).save();
			notifySaved();
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	public void reset() {
		try {
			File storeDir = box().archetype().repository().workorders().getStoreDirectory(order.store());
			if (storeDir.exists()) FileUtils.deleteDirectory(storeDir);
			form = new Form(formDefinition(), store());
			createFormulaCalculator();
			current = form.start(Language.valueOf(language()));
			refresh();
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private void notifySaved() {
		savedMessage.visible(true);
		Timer timer = new Timer();
		timer.schedule(new TimerTask() {
			@Override
			public void run() {
				hideSavedNotification();
			}
		}, 3000);
	}

	private void hideSavedNotification() {
		savedMessage.visible(false);
	}

	private void refreshProgress() {
		ProgressDisplay display = progress.display() != null ? progress.display() : createProgress();
		display.value(form.progress(), progressLabel());
		display.refresh();
	}

	private String progressLabel() {
		return null;
//        return form.progress() == 0 ? String.format(this.translate("%d steps"), form.definition.size()) : null;
	}

	private ProgressDisplay createProgress() {
		ProgressDisplay display = new ProgressDisplay(box());
		progress.display(display);
		return display;
	}

	private void createFormulaCalculator() {
		try {
			if (order == null) return;
			File template = WorkReportHelper.calculationTemplate(WorkReportGenerator.Archetype.OrderTypes.wrap(box().archetype().definitions().orderTypes()), order);
			if (!template.exists()) return;
			formulaCalculator = new WorkReportCalculator(template, OrderHelper.valuesOf(order, form));
			applyFormula();
		} catch (IOException | InvalidFormatException e) {
			Logger.error(e);
		}
	}

	private void applyFormula(Field field) {
		if (field == null || formulaCalculator == null) return;
		formulaCalculator.map(OrderHelper.valuesOf(order, form));
		Map<String, String> values = formulaCalculator.update(field.name(), FormHelper.valueOf(field.edition()));
		nonFieldValues(values).forEach(e -> form.store.put(e.getKey(), e.getValue()));
	}

	private void applyFormula() {
		if (formulaCalculator == null) return;
		Map<String, String> values = formulaCalculator.update();
		nonFieldValues(values).forEach(e -> form.store.put(e.getKey(), e.getValue()));
	}

	private Stream<Map.Entry<String, String>> nonFieldValues(Map<String, String> values) {
		return values.entrySet().stream().filter(e -> !isField(e.getKey()));
	}

	private boolean isField(String key) {
		return form.definition.fields().stream().anyMatch(f -> f.name.equalsIgnoreCase(key));
	}

}