package io.intino.konos.alexandria.rest;

import io.intino.konos.alexandria.exceptions.AlexandriaException;
import io.intino.konos.alexandria.rest.security.AlexandriaSecurityManager;
import io.intino.konos.alexandria.rest.security.NullSecurityManager;
import io.intino.konos.alexandria.rest.spark.PushService;
import io.intino.konos.alexandria.rest.spark.SparkManager;
import io.intino.konos.alexandria.rest.spark.SparkRouter;
import spark.Service;

import java.util.function.Consumer;
import java.util.function.Function;

public class AlexandriaSpark<R extends SparkRouter> {
	private AlexandriaSecurityManager securityManager = new NullSecurityManager();
	protected final String webDirectory;
	protected PushService pushService;
	protected static final String WebDirectory = "/www";
	protected Service service;
	protected int port;

	public AlexandriaSpark(int port) {
		this(port, WebDirectory);
	}

	public AlexandriaSpark(int port, String webDirectory) {
		this.port = port;
		this.webDirectory = webDirectory;
		this.service = Service.ignite();
		configureStaticFiles();
		service.port(this.port);
	}

	public AlexandriaSpark start() {
		service.init();
		return this;
	}

	public void stop() {
		if (service != null) service.stop();
		service = null;
	}

	public String webDirectory() {
		return webDirectory;
	}

	public void secure(AlexandriaSecurityManager manager) {
		this.securityManager = manager;
	}

	public R route(String path) {
		R router = createRouter(path);
		router.inject(pushService);
		router.whenRegisterPushService(pushServiceConsumer());
		router.whenValidate((Function<SparkManager<?>, Boolean>) manager -> securityManager.check(manager.fromQuery("hash", String.class), manager.fromQuery("signature", String.class)));
		return router;
	}

	private Consumer<PushService> pushServiceConsumer() {
		return (pushService) -> AlexandriaSpark.this.pushService = pushService;
	}

	protected R createRouter(String path) {
		return (R) new SparkRouter(service, path);
	}

	public interface ResourceCaller<SM extends SparkManager> {
		void call(SM manager) throws AlexandriaException;
	}

	protected void configureStaticFiles() {
		if (isInClasspath(webDirectory)) service.staticFileLocation(webDirectory);
		else service.externalStaticFileLocation(webDirectory);
	}

	private boolean isInClasspath(String path) {
		return getClass().getClassLoader().getResourceAsStream(path) != null;
	}
}
