package io.intino.cosmos.datahub.datamarts.master;

import io.intino.alexandria.Timetag;
import io.intino.alexandria.event.Event;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.terminal.Connector;
import io.intino.ness.master.reflection.*;
import io.intino.ness.master.model.Entity;
import io.intino.ness.master.Datamart.Translator;

import org.apache.activemq.command.ActiveMQTextMessage;

import java.io.File;
import java.util.*;
import java.io.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import java.util.stream.Collectors;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Instant;

import io.intino.cosmos.datahub.datamarts.master.*;

import static java.util.Objects.requireNonNull;

public class MasterDatamartImpl implements MasterDatamart {

	private static final String DATAHUB_MESSAGE_TOPIC = "service.ness.datamarts";
	private static final MasterDatamartImpl.MasterDatamartDefinition definition = new MasterDatamartImpl.MasterDatamartDefinition();

	private static final Set<String> TIMELINE_EVENTS = Set.of("Analysis","ComputerAssertion","Computer","Switch","ApplianceAssertion","CellularGateway","Wireless","Camera","ApplicationAssertion","Application","IncidentFinished","Incident");
	private static final Set<String> REEL_EVENTS = Set.of("Status");

	private final Connector connector;
	private final io.intino.cosmos.datahub.WizardTerminal.DatamartsRetryConfig retryConfig;
	private final List<EntityListener> entityListeners = new ArrayList<>();
	private final Map<String, List<MasterMounter>> mounters = new HashMap<>();
	private final MasterDatamart.Entities entities;
	private Instant ts;
	private boolean hasLocalAccessToTimelines;
	private final Map<String, TimelineNodeImpl> timelines = new java.util.concurrent.ConcurrentHashMap<>();
	private boolean hasLocalAccessToIndicators;
	private final Map<String, IndicatorNodeImpl> indicators = new java.util.concurrent.ConcurrentHashMap<>();
	private boolean hasLocalAccessToReels;
	private final Map<String, ReelNodeImpl> reels = new java.util.concurrent.ConcurrentHashMap<>();
	private volatile Translator translator = new Translator.Identity();
	private volatile boolean initializing = false;

	public MasterDatamartImpl(Connector connector, io.intino.cosmos.datahub.WizardTerminal.DatamartsRetryConfig retryConfig) {
		this.connector = requireNonNull(connector);
		this.retryConfig = requireNonNull(retryConfig);
		this.entities = new MasterDatamart.Entities(this);
		initMounters();
	}

	public synchronized MasterDatamartImpl init(String datamartSourceSelector) {
		try {
			initializing = true;
			downloadDatamartFromDatahub(datamartSourceSelector);
			Logger.info("MasterDatamart (" + (snapshotTimetag().isEmpty() ? "" : "snapshot " + snapshotTimetag() + ", ")  + connector.clientId() + ") initialized successfully.");
		} catch (Exception e) {
			throw new ExceptionInInitializerError("MasterDatamart failed to start because a " + e.getClass().getName() + " occurred: " + e.getMessage());
		} finally {
			initializing = false;
		}
		return this;
	}

	public boolean requiresDatahubNotifications() {
		return true;
	}

	@Override
	public int size() {
		return entities.size();
	}

	@Override
	public int sizeDisabled() {
		return entities.sizeDisabled();
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T extends Entity> T get(String id) {
		return (T) entities.get(id);
	}

	@Override
	public Stream<Entity> entities() {
		return entities.streamGeneric();
	}

	@Override
	public void addEntityListener(EntityListener listener) {
		if (listener == null) throw new NullPointerException("EntityListener cannot be null");
		entityListeners.add(listener);
	}

	@Override
	public DatamartDefinition getDefinition() {
		return definition;
	}

	@Override
    public List<String> listSnapshots() {
    	try {
    		jakarta.jms.Message message = requestResponseFromDatahub("listSnapshots", listSnapshotsRequest());
    		if (!message.getBooleanProperty("success")) throw new Exception(((jakarta.jms.TextMessage) message).getText());
    		return handleListSnapshotsResponse(message);
    	} catch (Exception e) {
    		Logger.error("Could not download list of available snapshots: " + e.getMessage(), e);
    		return java.util.Collections.emptyList();
    	}
    }

    private jakarta.jms.Message listSnapshotsRequest() throws Exception {
    	ActiveMQTextMessage message = new ActiveMQTextMessage();
    	message.setText("datamart=" + name() + ";operation=snapshots");
    	return message;
    }

    private List<String> handleListSnapshotsResponse(jakarta.jms.Message message) throws Exception {
    	return java.util.Arrays.stream(((jakarta.jms.TextMessage) message).getText().split(",")).collect(Collectors.toList());
    }

    @Override
    public synchronized MasterDatamart snapshot(String timetag) {
    	if (timetag == null) return this;
    	return new MasterDatamartImpl(connector, retryConfig) {
    		@Override
    		protected String snapshotTimetag() {
    			return timetag;
    		}
    		@Override
    		public synchronized MasterDatamart snapshot(String timetag) {
    			throw new java.lang.UnsupportedOperationException("Cannot request snapshots to snapshot instances of a datamart");
    		}
    	}.init(null);
	}

    public Instant ts() {
    	return this.ts;
    }

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Place place(String id) {
		return id == null ? null : entities.get(definition.placeEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Place placeDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.placeEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Place> places() {
		return entities.stream(definition.placeEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Place> placesDisabled() {
		return entities.streamDisabled(definition.placeEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Area area(String id) {
		return id == null ? null : entities.get(definition.areaEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Area areaDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.areaEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Area> areas() {
		return entities.stream(definition.areaEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Area> areasDisabled() {
		return entities.streamDisabled(definition.areaEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Tag tag(String id) {
		return id == null ? null : entities.get(definition.tagEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Tag tagDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.tagEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Tag> tags() {
		return entities.stream(definition.tagEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Tag> tagsDisabled() {
		return entities.streamDisabled(definition.tagEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule incidentRule(String id) {
		return id == null ? null : entities.get(definition.incidentRuleEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule incidentRuleDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.incidentRuleEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule> incidentRules() {
		return entities.stream(definition.incidentRuleEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule> incidentRulesDisabled() {
		return entities.streamDisabled(definition.incidentRuleEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Countermeasure countermeasure(String id) {
		return id == null ? null : entities.get(definition.countermeasureEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Countermeasure countermeasureDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.countermeasureEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Countermeasure> countermeasures() {
		return entities.stream(definition.countermeasureEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Countermeasure> countermeasuresDisabled() {
		return entities.streamDisabled(definition.countermeasureEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Model model(String id) {
		return id == null ? null : entities.get(definition.modelEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Model modelDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.modelEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Model> models() {
		return entities.stream(definition.modelEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Model> modelsDisabled() {
		return entities.streamDisabled(definition.modelEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.OrderType orderType(String id) {
		return id == null ? null : entities.get(definition.orderTypeEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.OrderType orderTypeDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.orderTypeEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.OrderType> orderTypes() {
		return entities.stream(definition.orderTypeEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.OrderType> orderTypesDisabled() {
		return entities.streamDisabled(definition.orderTypeEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Observable observable(String id) {
		return id == null ? null : entities.getDescendant(definition.observableEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Observable observableDisabled(String id) {
		return id == null ? null : entities.getDescendantDisabled(definition.observableEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Observable> observables() {
		return Stream.of(
			persons(),
			processes(),
			assets(),
			parts(),
			hardwares(),
			softwares()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Observable>flatMap(java.util.function.Function.identity());
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Observable> observablesDisabled() {
		return Stream.of(
			personsDisabled(),
			processesDisabled(),
			assetsDisabled(),
			partsDisabled(),
			hardwaresDisabled(),
			softwaresDisabled()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Observable>flatMap(java.util.function.Function.identity());
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Person person(String id) {
		return id == null ? null : entities.get(definition.personEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Person personDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.personEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Person> persons() {
		return entities.stream(definition.personEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Person> personsDisabled() {
		return entities.streamDisabled(definition.personEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Process process(String id) {
		return id == null ? null : entities.get(definition.processEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Process processDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.processEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Process> processes() {
		return entities.stream(definition.processEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Process> processesDisabled() {
		return entities.streamDisabled(definition.processEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Asset asset(String id) {
		return id == null ? null : entities.get(definition.assetEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Asset assetDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.assetEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Asset> assets() {
		return entities.stream(definition.assetEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Asset> assetsDisabled() {
		return entities.streamDisabled(definition.assetEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Part part(String id) {
		return id == null ? null : entities.get(definition.partEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Part partDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.partEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Part> parts() {
		return entities.stream(definition.partEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Part> partsDisabled() {
		return entities.streamDisabled(definition.partEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Hardware hardware(String id) {
		return id == null ? null : entities.getDescendant(definition.hardwareEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Hardware hardwareDisabled(String id) {
		return id == null ? null : entities.getDescendantDisabled(definition.hardwareEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Hardware> hardwares() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Hardware>stream(definition.hardwareEntityDefinition),
			appliances(),
			computers()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Hardware>flatMap(java.util.function.Function.identity());//.distinct();
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Hardware> hardwaresDisabled() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Hardware>streamDisabled(definition.hardwareEntityDefinition),
			appliancesDisabled(),
			computersDisabled()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Hardware>flatMap(java.util.function.Function.identity());
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Appliance appliance(String id) {
		return id == null ? null : entities.get(definition.applianceEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Appliance applianceDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.applianceEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Appliance> appliances() {
		return entities.stream(definition.applianceEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Appliance> appliancesDisabled() {
		return entities.streamDisabled(definition.applianceEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Computer computer(String id) {
		return id == null ? null : entities.get(definition.computerEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Computer computerDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.computerEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Computer> computers() {
		return entities.stream(definition.computerEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Computer> computersDisabled() {
		return entities.streamDisabled(definition.computerEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Software software(String id) {
		return id == null ? null : entities.getDescendant(definition.softwareEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Software softwareDisabled(String id) {
		return id == null ? null : entities.getDescendantDisabled(definition.softwareEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Software> softwares() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Software>stream(definition.softwareEntityDefinition),
			observers(),
			services(),
			applications()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Software>flatMap(java.util.function.Function.identity());//.distinct();
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Software> softwaresDisabled() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Software>streamDisabled(definition.softwareEntityDefinition),
			observersDisabled(),
			servicesDisabled(),
			applicationsDisabled()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Software>flatMap(java.util.function.Function.identity());
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Observer observer(String id) {
		return id == null ? null : entities.get(definition.observerEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Observer observerDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.observerEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Observer> observers() {
		return entities.stream(definition.observerEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Observer> observersDisabled() {
		return entities.streamDisabled(definition.observerEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Service service(String id) {
		return id == null ? null : entities.get(definition.serviceEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Service serviceDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.serviceEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Service> services() {
		return entities.stream(definition.serviceEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Service> servicesDisabled() {
		return entities.streamDisabled(definition.serviceEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Application application(String id) {
		return id == null ? null : entities.getDescendant(definition.applicationEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Application applicationDisabled(String id) {
		return id == null ? null : entities.getDescendantDisabled(definition.applicationEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Application> applications() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Application>stream(definition.applicationEntityDefinition),
			applicationJavas()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Application>flatMap(java.util.function.Function.identity());//.distinct();
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Application> applicationsDisabled() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Application>streamDisabled(definition.applicationEntityDefinition),
			applicationJavasDisabled()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Application>flatMap(java.util.function.Function.identity());
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.ApplicationJava applicationJava(String id) {
		return id == null ? null : entities.get(definition.applicationJavaEntityDefinition, id);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.ApplicationJava applicationJavaDisabled(String id) {
		return id == null ? null : entities.getDisabled(definition.applicationJavaEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.ApplicationJava> applicationJavas() {
		return entities.stream(definition.applicationJavaEntityDefinition);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.ApplicationJava> applicationJavasDisabled() {
		return entities.streamDisabled(definition.applicationJavaEntityDefinition);
	}

	@Override
	public Stream<TimelineNode> timelines(String id) {
		String key = id.replace(":", "-");
		Stream stream = timelines.values().stream().filter(t -> t.id.equals(key)).toList().stream();
		return stream;
	}
	@Override
	public TimelineNode analysisTimeline(String id) {
		String key = normalizedId(id, "Analysis");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "Analysis"));
		return TimelineNode.empty();
	}

	@Override
	public TimelineNode computerTimeline(String id) {
		String key = normalizedId(id, "Computer");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "Computer"));
		return TimelineNode.empty();
	}

	@Override
	public TimelineNode switchTimeline(String id) {
		String key = normalizedId(id, "Switch");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "Switch"));
		return TimelineNode.empty();
	}

	@Override
	public TimelineNode cellularGatewayTimeline(String id) {
		String key = normalizedId(id, "CellularGateway");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "CellularGateway"));
		return TimelineNode.empty();
	}

	@Override
	public TimelineNode wirelessTimeline(String id) {
		String key = normalizedId(id, "Wireless");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "Wireless"));
		return TimelineNode.empty();
	}

	@Override
	public TimelineNode cameraTimeline(String id) {
		String key = normalizedId(id, "Camera");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "Camera"));
		return TimelineNode.empty();
	}

	@Override
	public TimelineNode applicationTimeline(String id) {
		String key = normalizedId(id, "Application");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "Application"));
		return TimelineNode.empty();
	}

	@Override
	public TimelineNode incidentsTimeline(String id) {
		String key = normalizedId(id, "incidents");
		if (timelines.containsKey(key)) return timelines.get(key);
		if (!hasLocalAccessToTimelines) return timelines.computeIfAbsent(key, theId -> new TimelineNodeImpl(id, "incidents"));
		return TimelineNode.empty();
	}

	@Override
	public Stream<ReelNode> reels(String id) {
		String key = id.replace(":", "-");
		Stream stream = reels.values().stream().filter(r -> r.id.equals(key)).toList().stream();
		return stream;
	}
	@Override
	public ReelNode statusReel(String id) {
		String key = normalizedId(id, "Status");
		if (reels.containsKey(key)) return reels.get(key);
		if (!hasLocalAccessToReels) return reels.computeIfAbsent(key, theId -> new ReelNodeImpl(id, "Status"));
		return ReelNode.empty();
	}

	@Override
	public Stream<ReelNode> statusReels() {
		Stream stream = reels.values().stream().filter(r -> r.type.equals("Status")).toList().stream();
		return stream;
	}
	@Override
	public IndicatorNode pendingIncidentsIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("incidents.pendingIncidents") ? indicators.get("incidents.pendingIncidents") : IndicatorNode.empty();
		return indicators.computeIfAbsent("incidents.pendingIncidents", theId -> new IndicatorNodeImpl("incidents.pendingIncidents", null));
	}

	@Override
	public IndicatorNode incidentFinishedTimeIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("incidents.incidentFinishedTime") ? indicators.get("incidents.incidentFinishedTime") : IndicatorNode.empty();
		return indicators.computeIfAbsent("incidents.incidentFinishedTime", theId -> new IndicatorNodeImpl("incidents.incidentFinishedTime", null));
	}

	@Override
	public IndicatorNode availabilityIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Analysis.availability") ? indicators.get("Analysis.availability") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Analysis.availability", theId -> new IndicatorNodeImpl("Analysis.availability", null));
	}

	@Override
	public IndicatorNode healthIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Analysis.health") ? indicators.get("Analysis.health") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Analysis.health", theId -> new IndicatorNodeImpl("Analysis.health", null));
	}

	@Override
	public IndicatorNode riskIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Analysis.risk") ? indicators.get("Analysis.risk") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Analysis.risk", theId -> new IndicatorNodeImpl("Analysis.risk", null));
	}

	@Override
	public IndicatorNode usageRAMIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.usageRAM") ? indicators.get("Computer.usageRAM") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.usageRAM", theId -> new IndicatorNodeImpl("Computer.usageRAM", null));
	}

	@Override
	public IndicatorNode usageHDDIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.usageHDD") ? indicators.get("Computer.usageHDD") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.usageHDD", theId -> new IndicatorNodeImpl("Computer.usageHDD", null));
	}

	@Override
	public IndicatorNode usageCPUIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.usageCPU") ? indicators.get("Computer.usageCPU") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.usageCPU", theId -> new IndicatorNodeImpl("Computer.usageCPU", null));
	}

	@Override
	public IndicatorNode usageSystemIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.usageSystem") ? indicators.get("Computer.usageSystem") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.usageSystem", theId -> new IndicatorNodeImpl("Computer.usageSystem", null));
	}

	@Override
	public IndicatorNode usageThreadsIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.usageThreads") ? indicators.get("Computer.usageThreads") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.usageThreads", theId -> new IndicatorNodeImpl("Computer.usageThreads", null));
	}

	@Override
	public IndicatorNode usageFilesIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.usageFiles") ? indicators.get("Computer.usageFiles") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.usageFiles", theId -> new IndicatorNodeImpl("Computer.usageFiles", null));
	}

	@Override
	public IndicatorNode temperatureKernelIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.temperatureKernel") ? indicators.get("Computer.temperatureKernel") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.temperatureKernel", theId -> new IndicatorNodeImpl("Computer.temperatureKernel", null));
	}

	@Override
	public IndicatorNode temperatureExternalIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.temperatureExternal") ? indicators.get("Computer.temperatureExternal") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.temperatureExternal", theId -> new IndicatorNodeImpl("Computer.temperatureExternal", null));
	}

	@Override
	public IndicatorNode dataReceivedIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.dataReceived") ? indicators.get("Computer.dataReceived") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.dataReceived", theId -> new IndicatorNodeImpl("Computer.dataReceived", null));
	}

	@Override
	public IndicatorNode dataSentIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.dataSent") ? indicators.get("Computer.dataSent") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.dataSent", theId -> new IndicatorNodeImpl("Computer.dataSent", null));
	}

	@Override
	public IndicatorNode applicationsSystemIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.applicationsSystem") ? indicators.get("Computer.applicationsSystem") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.applicationsSystem", theId -> new IndicatorNodeImpl("Computer.applicationsSystem", null));
	}

	@Override
	public IndicatorNode applicationsKnownIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.applicationsKnown") ? indicators.get("Computer.applicationsKnown") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.applicationsKnown", theId -> new IndicatorNodeImpl("Computer.applicationsKnown", null));
	}

	@Override
	public IndicatorNode applicationsUnknownIndicator() {
		if (hasLocalAccessToIndicators) return indicators.containsKey("Computer.applicationsUnknown") ? indicators.get("Computer.applicationsUnknown") : IndicatorNode.empty();
		return indicators.computeIfAbsent("Computer.applicationsUnknown", theId -> new IndicatorNodeImpl("Computer.applicationsUnknown", null));
	}

	@Override
	public Translator translator() {
		return translator;
	}

	@Override
	public void translator(Translator translator) {
		this.translator = translator;
	}

	private void downloadDatamartFromDatahub(String datamartSourceSelector) throws DatahubRequestException {
		if (connector instanceof io.intino.alexandria.terminal.StubConnector) return;
		Logger.debug("Downloading datamart from datahub...");
		long start = java.lang.System.currentTimeMillis();

		Logger.debug("Downloading entities...");
		int[] eventCount = new int[1];
		loadEntitiesFromEvents(downloadEntities(eventCount, datamartSourceSelector), eventCount);

		loadTimelinesFromDatahub();
		loadIndicatorsFromDatahub();

		loadReelsFromDatahub();

		long time = java.lang.System.currentTimeMillis() - start;
		Logger.debug("Datamart downloaded from datahub after " + time + " ms");
	}

	private void loadTimelinesFromDatahub() {
		Logger.debug("Downloading timelines...");
		Boolean hasLocalAccessToTimelines = null;
	    for(String filename : listTimelineFiles()) {
	    	if (filename == null || filename.isBlank()) continue;
	    	try {
	           	File file = new File(filename);
	           	String id = file.getName().replace(".timeline", "");
	           	String type = file.getParentFile().getName();
	           	TimelineNodeImpl node = new TimelineNodeImpl(id, type, file.exists() ? file : null);
	           	timelines.put(id + ":" + type, node);
	           	if (hasLocalAccessToTimelines == null) hasLocalAccessToTimelines = file.exists();
	           	else hasLocalAccessToTimelines &= file.exists();
	        } catch (Exception e) {
	           	Logger.debug("Could not load timeline " + filename + " -> " + e.getClass().getSimpleName() + ": " + e.getMessage());
	        }
	    }
	    this.hasLocalAccessToTimelines = hasLocalAccessToTimelines != null && hasLocalAccessToTimelines;
	    Logger.debug("Loaded " + timelines.size() + " timelines (hasLocalAccessToTimelines=" + this.hasLocalAccessToTimelines + ")");
	}

	private String[] listTimelineFiles() {
		try {
			ActiveMQTextMessage request = new ActiveMQTextMessage();
			request.setText("datamart=" + name() + ";operation=list-timelines;");
			jakarta.jms.Message message = requestResponseFromDatahub("list-timelines", request);
			if (!message.getBooleanProperty("success")) throw new Exception(((jakarta.jms.TextMessage) message).getText());
			return ((jakarta.jms.TextMessage) message).getText().split(",");
		} catch (Exception e) {
			Logger.error(e);
			return new String[0];
		}
	}
	private void loadIndicatorsFromDatahub() {
		Logger.debug("Downloading indicators...");
		Boolean hasLocalAccessToIndicators = null;
	    for(String filename : listIndicatorFiles()) {
			if (filename == null || filename.isBlank()) continue;
			try {
				File file = new File(filename);
				String id = file.getName().replace(".indicator", "");
				IndicatorNodeImpl node = new IndicatorNodeImpl(id, file.exists() ? file : null);
				indicators.put(id, node);
				if (hasLocalAccessToIndicators == null) hasLocalAccessToIndicators = file.exists();
				else hasLocalAccessToIndicators &= file.exists();
			} catch (Exception e) {
				Logger.debug("Could not load indicator " + filename + " -> " + e.getClass().getSimpleName() + ": " + e.getMessage());
			}
		}
		this.hasLocalAccessToIndicators = hasLocalAccessToIndicators != null && hasLocalAccessToIndicators;
		Logger.debug("Loaded " + indicators.size() + " indicators (hasLocalAccessToTimelines=" + this.hasLocalAccessToTimelines + ")");
	}

	private String[] listIndicatorFiles() {
		try {
			ActiveMQTextMessage request = new ActiveMQTextMessage();
			request.setText("datamart=" + name() + ";operation=list-indicators;");
			jakarta.jms.Message message = requestResponseFromDatahub("list-indicators", request);
			if (!message.getBooleanProperty("success")) throw new Exception(((jakarta.jms.TextMessage) message).getText());
			return ((jakarta.jms.TextMessage) message).getText().split(",");
		} catch (Exception e) {
			Logger.error(e);
			return new String[0];
		}
	}

	private void loadReelsFromDatahub() {
		Logger.debug("Downloading reels...");
		Boolean hasLocalAccessToReels = null;
		for(String filename : listReelFiles()) {
			if (filename == null || filename.isBlank()) continue;
			try {
		    	File file = new File(filename);
		    	String id = file.getName().replace(".reel", "");
		    	String type = file.getParentFile().getName();
		    	ReelNodeImpl node = new ReelNodeImpl(id, type, file.exists() ? file : null);
		    	reels.put(id + ":" + type, node);
		    	if (hasLocalAccessToReels == null) hasLocalAccessToReels = file.exists();
	            else hasLocalAccessToReels &= file.exists();
		    } catch (Exception e) {
		    	Logger.debug("Could not load reel " + filename + " -> " + e.getClass().getSimpleName() + ": " + e.getMessage());
		    }
		}
		this.hasLocalAccessToReels = hasLocalAccessToReels != null && hasLocalAccessToReels;
		Logger.debug("Loaded " + reels.size() + " reels");
	}

	private String[] listReelFiles() {
		try {
			ActiveMQTextMessage request = new ActiveMQTextMessage();
			request.setText("datamart=" + name() + ";operation=list-reels;");
			jakarta.jms.Message message = requestResponseFromDatahub("list-reels", request);
			if (!message.getBooleanProperty("success")) throw new Exception(((jakarta.jms.TextMessage) message).getText());
			return ((jakarta.jms.TextMessage)message).getText().split(",");
		} catch (Exception e) {
			Logger.error(e);
			return new String[0];
		}
	}

	private Stream<Event> downloadEntities(int[] eventCount, String datamartSourceSelector) throws DatahubRequestException {
		try {
			jakarta.jms.Message message = requestResponseFromDatahub("downloadEvents", downloadEntitiesRequest(datamartSourceSelector));
			if (!message.getBooleanProperty("success")) throw new Exception(((jakarta.jms.TextMessage) message).getText());
			return handleDownloadResponse(message, eventCount);
		} catch (NullPointerException e) {
			throw new DatahubRequestException("Could not download datamart: no response from datahub.");
		} catch(DatahubRequestException e) {
			throw e;
		} catch (Exception e) {
			throw new DatahubRequestException("Could not download datamart: " + e.getMessage());
		}
	}

	private jakarta.jms.Message downloadEntitiesRequest(String datamartSourceSelector) throws Exception {
		ActiveMQTextMessage message = new ActiveMQTextMessage();
		message.setText("datamart=" + name() +
			";operation=entities" +
			(snapshotTimetag().isEmpty() ? "" : ";timetag=" + snapshotTimetag()) +
			(datamartSourceSelector == null ? "" : ";sourceSelector=" + datamartSourceSelector)
		);
		return message;
	}

	protected String snapshotTimetag() {
		return "";
	}

	private Stream<Event> handleDownloadResponse(jakarta.jms.Message message, int[] eventCount) throws Exception {
		jakarta.jms.BytesMessage m = (jakarta.jms.BytesMessage) message;
		eventCount[0] = m.getIntProperty("size");
		int size = m.getIntProperty("size");
		byte[] bytes = new byte[size];
		m.readBytes(bytes, size);
		return io.intino.alexandria.zim.ZimStream.of(new java.io.ByteArrayInputStream(bytes)).map(io.intino.alexandria.event.message.MessageEvent::new);
	}

	private void loadEntitiesFromEvents(Stream<Event> events, int[] eventCount) {
		events.forEach(this::mount);
	}

	public synchronized void mount(Event event) {
		try {
			if (event == null) return;
        	this.ts = event.ts();
        	mountEntities(event);
		} catch(Throwable e) {
			Logger.error("Error while mounting event ss=" + event.ss() + ", ts=" + event.ts() + " -> " + e.getMessage(), e);
		}
	}

	public synchronized void handleDatahubNotification(String notification) {
		try {
			if(notification == null || notification.isBlank()) return;
			String[] typeAndSs = notification.split("\0");
    		handleTimelineNotification(typeAndSs[0], typeAndSs[1]);
    		handleReelNotification(typeAndSs[0], typeAndSs[1]);
		} catch(Throwable e) {
			Logger.error("Error while processing datahub notification " + notification + ": " + e.getMessage(), e);
		}
	}

	private void handleReelNotification(String type, String id) {
		if (!REEL_EVENTS.contains(type)) return;
		String key = normalizedId(id, type);
		ReelNodeImpl reel = reels.computeIfAbsent(key, k -> new ReelNodeImpl(id, type, null));
		reel.notifyChange();
	}

	private void handleTimelineNotification(String type, String id) {
		if (!TIMELINE_EVENTS.contains(type)) return;
		String key = normalizedId(id, type);
		TimelineNodeImpl timeline = timelines.computeIfAbsent(key, k -> new TimelineNodeImpl(id, type, null));
		timeline.notifyChange();
	}

	private String normalizedId(String id, String type) {
		return id.replace(":", "-") + ":" + type;
	}

	private void mountEntities(Event event) {
		try {
			List<MasterMounter> mounters = this.mounters.get(event.type());
			if(mounters == null) return;
			mounters.forEach(mounter -> mounter.useListeners(!initializing).mount(event));
		} catch (Exception e) {
			Logger.error("Failed to mount event of type " + event.type() + ": " + e.getMessage(), e);
		}
	}

	private void initMounters() {
		mounters.computeIfAbsent("PlaceAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.PlaceMounter(entities, entityListeners));
		mounters.computeIfAbsent("AreaAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.AreaMounter(entities, entityListeners));
		mounters.computeIfAbsent("TagAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.TagMounter(entities, entityListeners));
		mounters.computeIfAbsent("IncidentRuleAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.IncidentRuleMounter(entities, entityListeners));
		mounters.computeIfAbsent("CountermeasureAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.CountermeasureMounter(entities, entityListeners));
		mounters.computeIfAbsent("ModelAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ModelMounter(entities, entityListeners));
		mounters.computeIfAbsent("OrderTypeAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.OrderTypeMounter(entities, entityListeners));
		mounters.computeIfAbsent("PersonAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.PersonMounter(entities, entityListeners));
		mounters.computeIfAbsent("ProcessAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ProcessMounter(entities, entityListeners));
		mounters.computeIfAbsent("AssetAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.AssetMounter(entities, entityListeners));
		mounters.computeIfAbsent("PartAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.PartMounter(entities, entityListeners));
		mounters.computeIfAbsent("HardwareAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.HardwareMounter(entities, entityListeners));
		mounters.computeIfAbsent("ApplianceAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ApplianceMounter(entities, entityListeners));
		mounters.computeIfAbsent("ComputerAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ComputerMounter(entities, entityListeners));
		mounters.computeIfAbsent("SoftwareAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.SoftwareMounter(entities, entityListeners));
		mounters.computeIfAbsent("ObserverAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ObserverMounter(entities, entityListeners));
		mounters.computeIfAbsent("ServiceAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ServiceMounter(entities, entityListeners));
		mounters.computeIfAbsent("ApplicationAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ApplicationMounter(entities, entityListeners));
		mounters.computeIfAbsent("ApplicationJavaAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ApplicationJavaMounter(entities, entityListeners));
	}

	private static int availableThreads() {
		return Runtime.getRuntime().availableProcessors();
	}

	private jakarta.jms.Message requestResponseFromDatahub(String requestName, jakarta.jms.Message request) throws DatahubRequestException {
    	long timeout = retryConfig.initialTimeoutAmount;
    	for(int i = 0; i < retryConfig.maxAttempts; i++) {
    		jakarta.jms.Message message = connector.requestResponse(DATAHUB_MESSAGE_TOPIC, request, timeout, retryConfig.timeoutUnit);
    		if (message != null) return message;
    		if (i < retryConfig.maxAttempts - 1) Logger.warn("("+(i+1)+") Datahub did not respond after " + timeout + " " + retryConfig.timeoutUnit + " to the request '" + requestName + "'. Trying again...");
    		timeout *= retryConfig.timeoutMultiplier;
    	}
    	throw new DatahubRequestException("Datahub did not respond to the request '" + requestName + "' after " + retryConfig.maxAttempts);
    }

	private class TimelineNodeImpl implements TimelineNode {
		private final String id;
		private final String type;
		private volatile File file;
		private volatile io.intino.sumus.chronos.TimelineStore timelineFile;
		private volatile ChangeListener listener;
		private volatile boolean disposed;

		private TimelineNodeImpl(String id, String type) {
			this(id, type, null);
		}

		private TimelineNodeImpl(String id, String type, File file) {
			this.id = requireNonNull(id);
			this.type = requireNonNull(type);
			this.file = file;
		}

		@Override
		public String id() {
			return id;
		}

		@Override
		public String type() {
			return type;
		}

		@Override
	    public boolean exists() {
	    	try {
	    		return timelineFile() != null;
	    	} catch(TimelineNotAvailableException e) {
	    		return false;
	    	}
	    }

		@Override
		public void dispose() {
			synchronized(this) {
				if (disposed) return;
				clearCache();
				listener = null;
				timelines.remove(id + ":" + type);
				disposed = true;
			}
		}

		@Override
		public io.intino.sumus.chronos.TimelineStore.TimeModel timeModel() throws TimelineNotAvailableException {
			return timelineFile().timeModel();
		}

		@Override
		public io.intino.sumus.chronos.TimelineStore.SensorModel sensorModel() throws TimelineNotAvailableException {
			return timelineFile().sensorModel();
		}

		@Override
		public Instant first() throws TimelineNotAvailableException {
			return timelineFile().first();
		}

		@Override
		public Instant last() throws TimelineNotAvailableException {
			return timelineFile().last();
		}

		@Override
		public io.intino.sumus.chronos.Timeline get() throws TimelineNotAvailableException {
			synchronized(this) {
				if (disposed) throw new TimelineNotAvailableException("This " + getClass().getSimpleName() + " is disposed.");
				try {
					return timelineFile().timeline();
				} catch(Exception e) {
					throw new TimelineNotAvailableException(e);
				}
			}
		}

		private io.intino.sumus.chronos.TimelineStore timelineFile() throws TimelineNotAvailableException {
			synchronized(this) {
				if (disposed) throw new TimelineNotAvailableException("This " + getClass().getSimpleName() + " is disposed.");
				try {
					if (hasLocalAccessToTimelines && file != null && file.exists()) return loadFile();
					if (timelineFile != null) return timelineFile;
					return timelineFile = downloadFromDatahub();
				} catch(Exception e) {
					throw new TimelineNotAvailableException(e);
				}
			}
		}

		@Override
		public void setChangeListener(ChangeListener listener) {
			this.listener = listener;
		}

		private void notifyChange() {
			synchronized(this) {
				if (disposed) return;
				try {
					clearCache();
					if (listener != null) new Thread(() -> listener.notifyChange(this), "Master-TimelineNodeImpl-" + System.currentTimeMillis()).start();
				} catch(Throwable ignored) {}
			}
		}

		private void clearCache() {
			timelineFile = null;
			if(!hasLocalAccessToTimelines) file = null;
		}

		private io.intino.sumus.chronos.TimelineStore loadFile() throws Exception {
			return io.intino.sumus.chronos.TimelineStore.of(file);
		}

		private io.intino.sumus.chronos.TimelineStore downloadFromDatahub() throws Exception {
			jakarta.jms.Message response = requestResponseFromDatahub("get-timeline=" + id(), request(hasLocalAccessToTimelines ? "path" : "download"));
			if (!response.getBooleanProperty("success")) throw new TimelineNotAvailableException("Could not get timeline " + id + " because datahub returned success=false in the response");;
			if (response instanceof jakarta.jms.TextMessage textResponse) {
				file = getFile(textResponse);
				hasLocalAccessToTimelines = true;
				if (file != null && file.exists()) return loadFile();
				file = null;
				hasLocalAccessToTimelines = false;
				response = requestResponseFromDatahub("get-timeline=" + id(), request("download"));
			}
			if (!response.getBooleanProperty("success")) throw new TimelineNotAvailableException("Could not get timeline " + id + " because datahub returned success=false in the response");;
			return readFromBytes((jakarta.jms.BytesMessage) response);
		}

		private io.intino.sumus.chronos.TimelineStore readFromBytes(jakarta.jms.BytesMessage m) throws Exception {
			int size = m.getIntProperty("size");
			byte[] bytes = new byte[size];
			m.readBytes(bytes, size);

			file = File.createTempFile(id(), ".timeline");
			java.nio.file.Files.write(file.toPath(), bytes, java.nio.file.StandardOpenOption.CREATE);
			file.deleteOnExit();
			hasLocalAccessToTimelines = false;

			return loadFile();
		}

		private File getFile(jakarta.jms.TextMessage m) {
			try {
				return new File(m.getText());
			} catch(Exception e) {
				return null;
			}
		}

		private jakarta.jms.Message request(String mode) throws Exception {
			ActiveMQTextMessage message = new ActiveMQTextMessage();
			String command = "datamart=" + name() + ";operation=get-timeline;id=" + id() + ";mode=" + mode + ";type=" + type;
			message.setText(command);
			return message;
		}
	}

	private class ReelNodeImpl implements ReelNode {
		private final String id;
		private final String type;
		private volatile File file;
		private volatile io.intino.sumus.chronos.ReelFile reelFile;
		private volatile ChangeListener listener;
		private volatile boolean disposed;

		private ReelNodeImpl(String id, String type) {
			this(id, type, null);
		}

		private ReelNodeImpl(String id, String type, File file) {
			this.id = requireNonNull(id);
			this.type = requireNonNull(type);
			this.file = file;
		}

		@Override
		public String id() {
			return id;
		}

		@Override
		public String type() {
			return type;
		}

		@Override
		public void dispose() {
			synchronized(this) {
				if (disposed) return;
				clearCache();
				listener = null;
				reels.remove(id + ":" + type);
				disposed = true;
			}
		}

		@Override
		public java.time.Instant start() throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			if (reelFile == null) return null;
			return reelFile.start();
		}

		@Override
		public io.intino.sumus.chronos.State stateOf(String signal) throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			return reelFile.lastStateOf(signal);
		}

		@Override
		public java.util.Set<String> signals() throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			return reelFile.signals();
		}

		@Override
		public List<io.intino.sumus.chronos.State> stateOf(Stream<String> signals) throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			return signals.map(reelFile::lastStateOf).toList();
		}

		public io.intino.sumus.chronos.Shot lastShotOf(String signal) throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			return reelFile.lastShotOf(signal);
		}

		public List<io.intino.sumus.chronos.Shot> lastShots() throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			return reelFile.lastShots();
		}

		public List<io.intino.sumus.chronos.Shot> lastShots(String group) throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			return reelFile.lastShots(group);
		}

		public List<io.intino.sumus.chronos.Shot> lastShots(io.intino.sumus.chronos.Group group) throws ReelNotAvailableException {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			return reelFile.lastShots(group);
		}

		@Override
	    public io.intino.sumus.chronos.Reel get(io.intino.sumus.chronos.Period period) throws ReelNotAvailableException {
	    	io.intino.sumus.chronos.ReelFile reelFile = reelFile();
	    	return reelFile().reel().by(period);
	    }

		@Override
	    public io.intino.sumus.chronos.Reel get(Instant from, Instant to, io.intino.sumus.chronos.Period period) throws ReelNotAvailableException {
	    	io.intino.sumus.chronos.ReelFile reelFile = reelFile();
	    	return reelFile.reel(from, to).by(period);
	    }

		@Override
	    public boolean exists() {
	    	try {
	    		return reelFile() != null;
	    	} catch(ReelNotAvailableException e) {
	    		return false;
	    	}
	    }

		private io.intino.sumus.chronos.ReelFile reelFile() throws ReelNotAvailableException {
			synchronized(this) {
				if (disposed) throw new ReelNotAvailableException("This " + getClass().getSimpleName() + " is disposed.");
				try {
					if (hasLocalAccessToReels && file != null && file.exists()) return loadFile();
					if (reelFile != null) return reelFile;
					return reelFile = downloadFromDatahub();
				} catch (Exception e) {
					throw new ReelNotAvailableException(e);
				}
			}
		}

		@Override
		public void setChangeListener(ChangeListener listener) {
			this.listener = listener;
		}

		private void notifyChange() {
			synchronized(this) {
				if (disposed) return;
				try {
					clearCache();
					if (listener != null) new Thread(() -> listener.notifyChange(this), "Master-ReelNodeImpl-" + System.currentTimeMillis()).start();
				} catch(Throwable ignored) {}
			}
		}

		private void clearCache() {
			reelFile = null;
			if(!hasLocalAccessToReels) file = null;
		}

		private io.intino.sumus.chronos.ReelFile loadFile() throws Exception {
			return io.intino.sumus.chronos.ReelFile.open(file);
		}

		private io.intino.sumus.chronos.ReelFile downloadFromDatahub() throws Exception {
			jakarta.jms.Message response = requestResponseFromDatahub("get-reel=" + id(), request(hasLocalAccessToReels ? "path" : "download"));
			if (!response.getBooleanProperty("success")) throw new ReelNotAvailableException("Could not get reel " + id + " because datahub returned success=false in the response");
			if (response instanceof jakarta.jms.TextMessage textResponse) {
				file = getFile(textResponse);
				hasLocalAccessToReels = true;
				if (file != null && file.exists()) return loadFile();
				file = null;
				hasLocalAccessToReels = false;
				response = requestResponseFromDatahub("get-reel=" + id(), request("download"));
			}
			if (!response.getBooleanProperty("success")) throw new ReelNotAvailableException("Could not get reel " + id + " because datahub returned success=false in the response");
			return readFromBytes((jakarta.jms.BytesMessage) response);
		}

		private io.intino.sumus.chronos.ReelFile readFromBytes(jakarta.jms.BytesMessage m) throws Exception {
			int size = m.getIntProperty("size");
			byte[] bytes = new byte[size];
			m.readBytes(bytes, size);
			file = File.createTempFile(id(), ".reel");
			java.nio.file.Files.write(file.toPath(), bytes, java.nio.file.StandardOpenOption.CREATE);
			file.deleteOnExit();
			hasLocalAccessToReels = false;
			return loadFile();
		}

		private File getFile(jakarta.jms.TextMessage m) {
			try {
				return new File(m.getText());
			} catch(Exception e) {
				return null;
			}
		}

		private jakarta.jms.Message request(String mode) throws Exception {
			ActiveMQTextMessage message = new ActiveMQTextMessage();
			String command = "datamart=" + name() + ";operation=get-reel;id=" + id() + ";mode=" + mode + ";type=" + type;
			message.setText(command);
			return message;
		}
	}

	private class IndicatorNodeImpl implements IndicatorNode {
		private final String id;
		private volatile File file;

		private IndicatorNodeImpl(String id, File file) {
			this.id = requireNonNull(id);
			this.file = file;
		}

		@Override
		public boolean exists() {
			try {
				return get() != null;
			} catch(IndicatorNotAvailableException e) {
				return false;
			}
		}

		@Override
		public Indicator get() throws IndicatorNotAvailableException {
			synchronized(this) {
				try {
					if (hasLocalAccessToIndicators) return Indicator.load(new java.io.FileInputStream(this.file));
					return downloadFromDatahub();
				} catch(Exception e) {
					throw new IndicatorNotAvailableException(e);
				}
			}
		}


		private Indicator downloadFromDatahub() throws Exception {
			jakarta.jms.Message response = requestResponseFromDatahub("get-indicator=" + id, request(hasLocalAccessToIndicators ? "path" : "download"));
	    	if (!response.getBooleanProperty("success")) throw new ReelNotAvailableException("Could not get indicator " + id + " because datahub returned success=false in the response");
	    	if (response instanceof jakarta.jms.TextMessage textResponse) {
	    		file = getFile(textResponse);
	    		hasLocalAccessToIndicators = true;
	    		if (file != null && file.exists()) return loadFile();
	    		file = null;
	    		hasLocalAccessToIndicators = false;
	    		response = requestResponseFromDatahub("get-indicator=" + id, request("download"));
	    	}
	    	if (!response.getBooleanProperty("success")) throw new ReelNotAvailableException("Could not get indicator " + id + " because datahub returned success=false in the response");
	    	return readFromBytes((jakarta.jms.BytesMessage) response);
		}

		private Indicator readFromBytes(jakarta.jms.BytesMessage m) throws Exception {
			int messageSize = m.getIntProperty("size");
			byte[] bytes = new byte[messageSize];
			m.readBytes(bytes, messageSize);
			return Indicator.load(new ByteArrayInputStream(bytes));
		}

		private jakarta.jms.Message request(String mode) throws Exception {
			ActiveMQTextMessage message = new ActiveMQTextMessage();
			message.setText("datamart=" + name() + ";operation=get-indicator;id=" + id + ";mode=" + mode);
			return message;
		}

		private File getFile(jakarta.jms.TextMessage m) {
			try {
				return new File(m.getText());
			} catch(Exception e) {
				return null;
			}
		}

		private Indicator loadFile() throws Exception {
			return Indicator.load(new java.io.FileInputStream(this.file));
		}
	}

	private static Set<String> sourcesOfTimeline(String type) {
    	return switch(type) {
    		case "Analysis" -> Set.of("Analysis");
    		case "Computer" -> Set.of("Computer","ComputerAssertion");
    		case "Switch" -> Set.of("Switch","ApplianceAssertion");
    		case "CellularGateway" -> Set.of("CellularGateway","ApplianceAssertion");
    		case "Wireless" -> Set.of("Wireless","ApplianceAssertion");
    		case "Camera" -> Set.of("Camera","ApplianceAssertion");
    		case "Application" -> Set.of("Application","ApplicationAssertion");
    		case "incidents" -> Set.of("IncidentFinished","Incident");
    		default -> java.util.Collections.emptySet();
    	};
    }

	private static Set<String> sourcesOfReel(String type) {
    	return switch(type) {
    		case "Status" -> Set.of("Datalake#dl$status");
    		default -> java.util.Collections.emptySet();
    	};
    }

	public static final class MasterDatamartDefinition implements DatamartDefinition {

		@Override
		public String name() {
			return "master";
		}

		@Override
		public Scale scale() {
			return Scale.Day;
		}

		@Override
		public Query<EntityDefinition> entities() {
			return new Query<EntityDefinition>(List.of(placeEntityDefinition,areaEntityDefinition,tagEntityDefinition,incidentRuleEntityDefinition,countermeasureEntityDefinition,modelEntityDefinition,orderTypeEntityDefinition,observableEntityDefinition,personEntityDefinition,processEntityDefinition,assetEntityDefinition,partEntityDefinition,hardwareEntityDefinition,applianceEntityDefinition,computerEntityDefinition,softwareEntityDefinition,observerEntityDefinition,serviceEntityDefinition,applicationEntityDefinition,applicationJavaEntityDefinition));
		}

		@Override
		public Query<StructDefinition> structs() {
			return new Query<StructDefinition>(List.of(countermeasure$ActuationStructDefinition,model$ProfileStructDefinition,model$Profile$VariableStructDefinition,observable$OperationStructDefinition,observable$Operation$ProcedureStructDefinition,observable$Operation$Procedure$ParameterStructDefinition));
		}

		@Override
		public Optional<EntityDefinition> entity(String name) {
			switch(name) {
				case "Place": return Optional.of(placeEntityDefinition);
				case "Area": return Optional.of(areaEntityDefinition);
				case "Tag": return Optional.of(tagEntityDefinition);
				case "IncidentRule": return Optional.of(incidentRuleEntityDefinition);
				case "Countermeasure": return Optional.of(countermeasureEntityDefinition);
				case "Model": return Optional.of(modelEntityDefinition);
				case "OrderType": return Optional.of(orderTypeEntityDefinition);
				case "Observable": return Optional.of(observableEntityDefinition);
				case "Person": return Optional.of(personEntityDefinition);
				case "Process": return Optional.of(processEntityDefinition);
				case "Asset": return Optional.of(assetEntityDefinition);
				case "Part": return Optional.of(partEntityDefinition);
				case "Hardware": return Optional.of(hardwareEntityDefinition);
				case "Appliance": return Optional.of(applianceEntityDefinition);
				case "Computer": return Optional.of(computerEntityDefinition);
				case "Software": return Optional.of(softwareEntityDefinition);
				case "Observer": return Optional.of(observerEntityDefinition);
				case "Service": return Optional.of(serviceEntityDefinition);
				case "Application": return Optional.of(applicationEntityDefinition);
				case "ApplicationJava": return Optional.of(applicationJavaEntityDefinition);
			}
			return Optional.empty();
		}

		@Override
    	public Optional<StructDefinition> struct(String name) {
    		switch(name) {
    			case "Countermeasure$Actuation": return Optional.of(countermeasure$ActuationStructDefinition);
    			case "Model$Profile": return Optional.of(model$ProfileStructDefinition);
    			case "Model$Profile$Variable": return Optional.of(model$Profile$VariableStructDefinition);
    			case "Observable$Operation": return Optional.of(observable$OperationStructDefinition);
    			case "Observable$Operation$Procedure": return Optional.of(observable$Operation$ProcedureStructDefinition);
    			case "Observable$Operation$Procedure$Parameter": return Optional.of(observable$Operation$Procedure$ParameterStructDefinition);
    		}
    		return Optional.empty();
    	}

		private MasterDatamartDefinition datamart() {
			return this;
		}

		public final EntityDefinition placeEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Place";}
			public String name() {return "Place";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Place.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "label";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "coordinates";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "timeOffset";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "region";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "city";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "area";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Area.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "address";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "postalCode";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition areaEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Area";}
			public String name() {return "Area";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Area.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "label";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition tagEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Tag";}
			public String name() {return "Tag";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Tag.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition incidentRuleEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "IncidentRule";}
			public String name() {return "IncidentRule";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "target";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "model";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "condition";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "precondition";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "level";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.Level.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "tracking";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "dismissCondition";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "description";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "triggers";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition countermeasureEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Countermeasure";}
			public String name() {return "Countermeasure";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Countermeasure.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "condition";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "priority";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "actuationList";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().countermeasure$ActuationStructDefinition);}
		    				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Countermeasure.Actuation.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition modelEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Model";}
			public String name() {return "Model";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "type";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "target";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "profile";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition orderTypeEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "OrderType";}
			public String name() {return "OrderType";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.OrderType.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition observableEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable";}
			public String name() {return "Observable";}
			public boolean isAbstract() {return true;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of(personEntityDefinition,processEntityDefinition,assetEntityDefinition,partEntityDefinition,hardwareEntityDefinition,applianceEntityDefinition,computerEntityDefinition,softwareEntityDefinition,observerEntityDefinition,serviceEntityDefinition,applicationEntityDefinition,applicationJavaEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "label";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "place";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Place.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "team";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "model";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "container";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "observer";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Observer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "responsible";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Person.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "tags";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "operationList";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().observable$OperationStructDefinition);}
		    				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition personEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Person";}
			public String name() {return "Person";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Person.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "area";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Area.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "role";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "supervisor";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Person.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "email";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "phone";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition processEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Process";}
			public String name() {return "Process";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Process.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition assetEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Asset";}
			public String name() {return "Asset";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Asset.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition partEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Part";}
			public String name() {return "Part";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Part.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition hardwareEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Hardware";}
			public String name() {return "Hardware";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of(applianceEntityDefinition,computerEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Hardware.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "sn";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "ip";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "publicIp";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition applianceEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Hardware.Appliance";}
			public String name() {return "Appliance";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(hardwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(hardwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Appliance.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition computerEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Hardware.Computer";}
			public String name() {return "Computer";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(hardwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(hardwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Computer.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "architecture";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "os";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "cores";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "diskSize";}
		    		public Class<?> type() {return Long.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "memorySize";}
		    		public Class<?> type() {return Long.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "processes";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "maxOpenFiles";}
		    		public Class<?> type() {return Long.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jvm";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "isp";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition softwareEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software";}
			public String name() {return "Software";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of(observerEntityDefinition,serviceEntityDefinition,applicationEntityDefinition,applicationJavaEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Software.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition observerEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Observer";}
			public String name() {return "Observer";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(softwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observer.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "version";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "installedActivities";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "enabledActivities";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition serviceEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Service";}
			public String name() {return "Service";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(softwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Service.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "url";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition applicationEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Application";}
			public String name() {return "Application";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(softwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of(applicationJavaEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Application.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "name";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "user";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "systemService";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "startingTime";}
		    		public Class<?> type() {return LocalDateTime.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "pid";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "commandLine";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "state";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Application.State.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition applicationJavaEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Application.ApplicationJava";}
			public String name() {return "ApplicationJava";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(applicationEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(applicationEntityDefinition,softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.ApplicationJava.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "classpathPrefix";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "mainArtifact";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "debugPort";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jmxPort";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "minMemory";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "maxMemory";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "classpath";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "inputArguments";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jvmParameter";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jvmVersion";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if (other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};

        public final StructDefinition countermeasure$ActuationStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Countermeasure$Actuation";}
        	public String name() {return "Actuation";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Countermeasure.Actuation.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "id";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "label";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "target";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "tries";}
            		public Class<?> type() {return Integer.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "tryInterval";}
            		public Class<?> type() {return Integer.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "extraCondition";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "notification";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "dismissNotification";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "channels";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
            				public Class<?> javaClass() {return String.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if (other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition model$ProfileStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Model$Profile";}
        	public String name() {return "Profile";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "variableList";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().model$Profile$VariableStructDefinition);}
            				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.Variable.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if (other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition model$Profile$VariableStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Model$Profile$Variable";}
        	public String name() {return "Variable";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.Variable.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "className";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "label";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "type";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "operator";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "unit";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "symbol";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "values";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
            				public Class<?> javaClass() {return String.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "priority";}
            		public Class<?> type() {return Integer.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "min";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "max";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "description";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "format";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if (other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition observable$OperationStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Observable$Operation";}
        	public String name() {return "Operation";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "activity";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "procedureList";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().observable$Operation$ProcedureStructDefinition);}
            				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if (other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition observable$Operation$ProcedureStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Observable$Operation$Procedure";}
        	public String name() {return "Procedure";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "returnType";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "description";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "parameterList";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().observable$Operation$Procedure$ParameterStructDefinition);}
            				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.Parameter.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if (other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition observable$Operation$Procedure$ParameterStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Observable$Operation$Procedure$Parameter";}
        	public String name() {return "Parameter";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.Parameter.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "type";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if (other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
	}

	static {
		try {
			Object ref = MasterDatamart.class.getDeclaredField("definition").get(null);
			java.lang.reflect.Field field = ref.getClass().getDeclaredField("definition");
			field.setAccessible(true);
			field.set(ref, definition);
			field.setAccessible(false);
		} catch (Exception e) {
			throw new ExceptionInInitializerError("Could not set MasterDatamart.definition field");
		}
	}
}