package io.intino.cosmos.datahub.datamarts.master;

import io.intino.alexandria.Timetag;
import io.intino.alexandria.event.Event;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.terminal.Connector;
import io.intino.ness.master.reflection.*;
import io.intino.ness.master.model.Entity;

import org.apache.activemq.command.ActiveMQTextMessage;

import java.io.File;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import java.util.stream.Collectors;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Instant;

import io.intino.cosmos.datahub.datamarts.master.*;

import static java.util.Objects.requireNonNull;

public class MasterDatamartImpl implements MasterDatamart {

	private static final String DATAHUB_MESSAGE_TOPIC = "service.ness.datamarts";
	private static final MasterDatamartImpl.MasterDatamartDefinition definition = new MasterDatamartImpl.MasterDatamartDefinition();

	private static final Set<String> TIMELINE_EVENTS = Set.of("Computer","Switch","CellularGateway","Wireless","Camera","Application");
	private static final Set<String> REEL_EVENTS = Set.of("status");

	private final Connector connector;
	private final io.intino.cosmos.datahub.WizardTerminal.DatamartsRetryConfig retryConfig;
	private final AtomicBoolean initialized = new AtomicBoolean(false);
	private final List<EntityListener> entityListeners = new ArrayList<>();
	private final Map<String, List<MasterMounter>> mounters = new HashMap<>();
	private final MasterDatamart.Entities entities;
	private Instant ts;
	private final Map<String, TimelineNodeImpl> timelines = new java.util.concurrent.ConcurrentHashMap<>();
	private final Map<String, ReelNodeImpl> reels = new java.util.concurrent.ConcurrentHashMap<>();

	public MasterDatamartImpl(Connector connector, io.intino.cosmos.datahub.WizardTerminal.DatamartsRetryConfig retryConfig) {
		this.connector = requireNonNull(connector);
		this.retryConfig = requireNonNull(retryConfig);
		this.entities = new MasterDatamart.Entities(this);
		initMounters();
	}

	public synchronized MasterDatamartImpl init(String datamartSourceSelector) {
		try {
			if (!initialized.compareAndSet(false, true)) return this;
			downloadDatamartFromDatahub(datamartSourceSelector);
			Logger.info("MasterDatamart (" + (snapshotTimetag().isEmpty() ? "" : "snapshot " + snapshotTimetag() + ", ")  + connector.clientId() + ") initialized successfully.");
		} catch (Exception e) {
			throw new ExceptionInInitializerError("MasterDatamart failed to start because a " + e.getClass().getName() + " occurred: " + e.getMessage());
		}
		return this;
	}

	@Override
	public int size() {
		return entities.size();
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T extends Entity> T get(String id) {
		return (T) entities.get(id);
	}

	@Override
	public Stream<Entity> entities() {
		return entities.streamGeneric();
	}

	@Override
	public void addEntityListener(EntityListener listener) {
		if(listener == null) throw new NullPointerException("EntityListener cannot be null");
		entityListeners.add(listener);
	}

	@Override
	public DatamartDefinition getDefinition() {
		return definition;
	}

	@Override
    public List<String> listSnapshots() {
    	try {
    		javax.jms.Message message = requestResponseFromDatahub("listSnapshots", listSnapshotsRequest());
    		if (!message.getBooleanProperty("success")) throw new Exception(((javax.jms.TextMessage) message).getText());
    		return handleListSnapshotsResponse(message);
    	} catch (Exception e) {
    		Logger.error("Could not download list of available snapshots: " + e.getMessage(), e);
    		return java.util.Collections.emptyList();
    	}
    }

    private javax.jms.Message listSnapshotsRequest() throws Exception {
    	ActiveMQTextMessage message = new ActiveMQTextMessage();
    	message.setText("datamart=" + name() + ";operation=snapshots");
    	return message;
    }

    private List<String> handleListSnapshotsResponse(javax.jms.Message message) throws Exception {
    	return java.util.Arrays.stream(((javax.jms.TextMessage) message).getText().split(",")).collect(Collectors.toList());
    }

    @Override
    public synchronized MasterDatamart snapshot(String timetag) {
    	if(timetag == null) return this;
    	return new MasterDatamartImpl(connector, retryConfig) {
    		@Override
    		protected String snapshotTimetag() {
    			return timetag;
    		}
    		@Override
    		public synchronized MasterDatamart snapshot(String timetag) {
    			throw new java.lang.UnsupportedOperationException("Cannot request snapshots to snapshot instances of a datamart");
    		}
    	}.init(null);
    }

    public Instant ts() {
    	return this.ts;
    }

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Place place(String id) {
		return id == null ? null : entities.get(definition.placeEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Place> places() {
		return entities.stream(definition.placeEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Area area(String id) {
		return id == null ? null : entities.get(definition.areaEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Area> areas() {
		return entities.stream(definition.areaEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule incidentRule(String id) {
		return id == null ? null : entities.get(definition.incidentRuleEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule> incidentRules() {
		return entities.stream(definition.incidentRuleEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Model model(String id) {
		return id == null ? null : entities.get(definition.modelEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Model> models() {
		return entities.stream(definition.modelEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.OrderType orderType(String id) {
		return id == null ? null : entities.get(definition.orderTypeEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.OrderType> orderTypes() {
		return entities.stream(definition.orderTypeEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Observable observable(String id) {
		return id == null ? null : entities.getDescendant(definition.observableEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Observable> observables() {
		return Stream.of(
			persons(),
			processes(),
			assets(),
			parts(),
			hardwares(),
			softwares()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Observable>flatMap(java.util.function.Function.identity());//.distinct();
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Person person(String id) {
		return id == null ? null : entities.get(definition.personEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Person> persons() {
		return entities.stream(definition.personEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Process process(String id) {
		return id == null ? null : entities.get(definition.processEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Process> processes() {
		return entities.stream(definition.processEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Asset asset(String id) {
		return id == null ? null : entities.get(definition.assetEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Asset> assets() {
		return entities.stream(definition.assetEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Part part(String id) {
		return id == null ? null : entities.get(definition.partEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Part> parts() {
		return entities.stream(definition.partEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Hardware hardware(String id) {
		return id == null ? null : entities.getDescendant(definition.hardwareEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Hardware> hardwares() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Hardware>stream(definition.hardwareEntityDefinition),
			appliances(),
			computers()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Hardware>flatMap(java.util.function.Function.identity());//.distinct();
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Appliance appliance(String id) {
		return id == null ? null : entities.get(definition.applianceEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Appliance> appliances() {
		return entities.stream(definition.applianceEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Computer computer(String id) {
		return id == null ? null : entities.get(definition.computerEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Computer> computers() {
		return entities.stream(definition.computerEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Software software(String id) {
		return id == null ? null : entities.getDescendant(definition.softwareEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Software> softwares() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Software>stream(definition.softwareEntityDefinition),
			observers(),
			services(),
			applications()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Software>flatMap(java.util.function.Function.identity());//.distinct();
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Observer observer(String id) {
		return id == null ? null : entities.get(definition.observerEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Observer> observers() {
		return entities.stream(definition.observerEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Service service(String id) {
		return id == null ? null : entities.get(definition.serviceEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Service> services() {
		return entities.stream(definition.serviceEntityDefinition);
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.Application application(String id) {
		return id == null ? null : entities.getDescendant(definition.applicationEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.Application> applications() {
		return Stream.of(
			entities.<io.intino.cosmos.datahub.datamarts.master.entities.Application>stream(definition.applicationEntityDefinition),
			javaApplications()
		).<io.intino.cosmos.datahub.datamarts.master.entities.Application>flatMap(java.util.function.Function.identity());//.distinct();
	}

	@Override
	public io.intino.cosmos.datahub.datamarts.master.entities.JavaApplication javaApplication(String id) {
		return id == null ? null : entities.get(definition.javaApplicationEntityDefinition, id);
	}

	@Override
	public Stream<io.intino.cosmos.datahub.datamarts.master.entities.JavaApplication> javaApplications() {
		return entities.stream(definition.javaApplicationEntityDefinition);
	}

	@Override
	public Stream<TimelineNode> timelines(String id) {
		Stream stream = timelines.values().stream().filter(t -> t.id.equals(id)).toList().stream();
		return stream;
	}
	@Override
	public TimelineNode computerTimeline(String id) {
		return timelines.computeIfAbsent(id + ":Computer", theId -> new TimelineNodeImpl(id, "Computer", Set.of("Computer","ComputerAssertion")));
	}

	@Override
	public TimelineNode switchTimeline(String id) {
		return timelines.computeIfAbsent(id + ":Switch", theId -> new TimelineNodeImpl(id, "Switch", Set.of("Switch","ApplianceAssertion")));
	}

	@Override
	public TimelineNode cellularGatewayTimeline(String id) {
		return timelines.computeIfAbsent(id + ":CellularGateway", theId -> new TimelineNodeImpl(id, "CellularGateway", Set.of("CellularGateway","ApplianceAssertion")));
	}

	@Override
	public TimelineNode wirelessTimeline(String id) {
		return timelines.computeIfAbsent(id + ":Wireless", theId -> new TimelineNodeImpl(id, "Wireless", Set.of("Wireless","ApplianceAssertion")));
	}

	@Override
	public TimelineNode cameraTimeline(String id) {
		return timelines.computeIfAbsent(id + ":Camera", theId -> new TimelineNodeImpl(id, "Camera", Set.of("Camera","ApplianceAssertion")));
	}

	@Override
	public TimelineNode applicationTimeline(String id) {
		return timelines.computeIfAbsent(id + ":Application", theId -> new TimelineNodeImpl(id, "Application", Set.of("Application","ApplicationAssertion")));
	}

	@Override
	public Stream<ReelNode> reels(String id) {
		Stream stream = reels.values().stream().filter(r -> r.id.equals(id)).toList().stream();
		return stream;
	}
	@Override
	public ReelNode statusReel(String id) {
		return reels.computeIfAbsent(id + ":Status", theId -> new ReelNodeImpl(id, "Status", Set.of("Datalake#dl$status")));
	}

	@Override
	public Stream<ReelNode> statusReels() {
		Stream stream = reels.values().stream().filter(r -> r.type.equals("Status")).toList().stream();
		return stream;
	}

	private void downloadDatamartFromDatahub(String datamartSourceSelector) {
		if (connector instanceof io.intino.alexandria.terminal.StubConnector) return;
		Logger.debug("Downloading datamart from datahub...");
		long start = java.lang.System.currentTimeMillis();

		Logger.debug("Downloading entities...");
		int[] eventCount = new int[1];
		loadEntitiesFromEvents(downloadEntities(eventCount, datamartSourceSelector), eventCount);

		loadTimelinesFromDatahub();

		loadReelsFromDatahub();

		long time = java.lang.System.currentTimeMillis() - start;
		Logger.debug("Datamart downloaded from datahub after " + time + " ms");
	}

	private void loadTimelinesFromDatahub() {
		Logger.debug("Downloading timelines...");
	    for(String filename : listTimelineFiles()) {
	    	if(filename == null || filename.isBlank()) continue;
	    	try {
	           	File file = new File(filename);
	           	String id = file.getName().replace(".timeline", "");
	           	String type = file.getParentFile().getName();
	           	TimelineNodeImpl node = new TimelineNodeImpl(id, type, sourcesOfTimeline(type), file.exists() ? file : null);
	           	timelines.put(id + ":" + type, node);
	        } catch (Exception e) {
	           	Logger.debug("Could not load timeline " + filename + " -> " + e.getClass().getSimpleName() + ": " + e.getMessage());
	        }
	    }
	    Logger.debug("Loaded " + timelines.size() + " timelines");
	}

	private String[] listTimelineFiles() {
		try {
			ActiveMQTextMessage request = new ActiveMQTextMessage();
			request.setText("datamart=" + name() + ";operation=list-timelines;");
			javax.jms.Message message = requestResponseFromDatahub("list-timelines", request);
			if (!message.getBooleanProperty("success")) throw new Exception(((javax.jms.TextMessage) message).getText());
			return ((javax.jms.TextMessage)message).getText().split(",");
		} catch (Exception e) {
			Logger.error(e);
			return new String[0];
		}
	}

	private void loadReelsFromDatahub() {
		Logger.debug("Downloading reels...");
		for(String filename : listReelFiles()) {
			if(filename == null || filename.isBlank()) continue;
			try {
		    	File file = new File(filename);
		    	String id = file.getName().replace(".reel", "");
		    	String type = file.getParentFile().getName();
		    	ReelNodeImpl node = new ReelNodeImpl(id, type, sourcesOfReel(type), file.exists() ? file : null);
		    	reels.put(id + ":" + type, node);
		    } catch (Exception e) {
		    	Logger.debug("Could not load reel " + filename + " -> " + e.getClass().getSimpleName() + ": " + e.getMessage());
		    }
		}
		Logger.debug("Loaded " + reels.size() + " reels");
	}

	private String[] listReelFiles() {
		try {
			ActiveMQTextMessage request = new ActiveMQTextMessage();
			request.setText("datamart=" + name() + ";operation=list-reels;");
			javax.jms.Message message = requestResponseFromDatahub("list-reels", request);
			if (!message.getBooleanProperty("success")) throw new Exception(((javax.jms.TextMessage) message).getText());
			return ((javax.jms.TextMessage)message).getText().split(",");
		} catch (Exception e) {
			Logger.error(e);
			return new String[0];
		}
	}

	private Stream<Event> downloadEntities(int[] eventCount, String datamartSourceSelector) {
		try {
			javax.jms.Message message = requestResponseFromDatahub("downloadEvents", downloadEntitiesRequest(datamartSourceSelector));
			if (!message.getBooleanProperty("success")) throw new Exception(((javax.jms.TextMessage) message).getText());
			return handleDownloadResponse(message, eventCount);
		} catch (NullPointerException e) {
			throw new NullPointerException("Could not download datamart: no response from datahub.");
		} catch (Exception e) {
			throw new RuntimeException("Could not download datamart: " + e.getMessage());
		}
	}

	private javax.jms.Message downloadEntitiesRequest(String datamartSourceSelector) throws Exception {
		ActiveMQTextMessage message = new ActiveMQTextMessage();
		message.setText("datamart=" + name() +
			";operation=entities" +
			(snapshotTimetag().isEmpty() ? "" : ";timetag=" + snapshotTimetag()) +
			(datamartSourceSelector == null ? "" : ";sourceSelector=" + datamartSourceSelector)
		);
		return message;
	}

	protected String snapshotTimetag() {
		return "";
	}

	private Stream<Event> handleDownloadResponse(javax.jms.Message message, int[] eventCount) throws Exception {
		javax.jms.BytesMessage m = (javax.jms.BytesMessage) message;
		eventCount[0] = m.getIntProperty("size");
		int size = m.getIntProperty("size");
		byte[] bytes = new byte[size];
		m.readBytes(bytes, size);
		return io.intino.alexandria.zim.ZimStream.of(new java.io.ByteArrayInputStream(bytes)).map(io.intino.alexandria.event.message.MessageEvent::new);
	}

	private void loadEntitiesFromEvents(Stream<Event> events, int[] eventCount) {
		events.forEach(this::mount);
	}

	public synchronized void mount(Event event) {
		if(event == null) return;
		this.ts = event.ts();
		mountEntities(event);
		mountTimelines(event);
		mountReels(event);
	}

	private void mountReels(Event event) {
		if(!REEL_EVENTS.contains(event.type())) return;
		String id = sensorParameter(event.ss());
		String type = event.type();
		String key = id + ":" + type;
		if(!reels.containsKey(key)) {
			ReelNodeImpl node = new ReelNodeImpl(id, type, sourcesOfReel(type), null);
			reels.put(key, node);
		}
		reels.values().forEach(reel -> reel.notifyEvent(event));
	}

	private void mountTimelines(Event event) {
		if(!TIMELINE_EVENTS.contains(event.type())) return;
		String id = sensorParameter(event.ss());
		String type = event.type();
		TimelineNodeImpl node = timelines.computeIfAbsent(id + ":" + type, k -> new TimelineNodeImpl(id, type, sourcesOfTimeline(type), null));
		node.notifyEvent(event);
	}

	private static String sensorParameter(String ss) {
		if (ss.contains("?")) {
			try {
				Map<String, String> map = Arrays.stream(ss.substring(ss.indexOf("?") + 1).split(";"))
						.map(p -> p.split("="))
						.collect(Collectors.toMap(f -> f[0], f -> f[1]));
				if (map.containsKey("sensor")) return map.get("sensor");
				else ss.substring(0, ss.indexOf("?"));
			} catch (Exception e) {
				Logger.error(e);
			}
		}
		return ss;
	}

	private void mountEntities(Event event) {
		try {
			java.util.Optional.ofNullable(this.mounters.get(event.type())).ifPresent(mounters -> mounters.forEach(mounter -> mounter.mount(event)));
		} catch (Exception e) {
			Logger.error("Failed to mount event of type " + event.type() + ": " + e.getMessage(), e);
		}
	}

	private void initMounters() {
		mounters.computeIfAbsent("PlaceAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.PlaceMounter(entities, entityListeners));
		mounters.computeIfAbsent("AreaAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.AreaMounter(entities, entityListeners));
		mounters.computeIfAbsent("IncidentRuleAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.IncidentRuleMounter(entities, entityListeners));
		mounters.computeIfAbsent("ModelAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ModelMounter(entities, entityListeners));
		mounters.computeIfAbsent("OrderTypeAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.OrderTypeMounter(entities, entityListeners));
		mounters.computeIfAbsent("PersonAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.PersonMounter(entities, entityListeners));
		mounters.computeIfAbsent("ProcessAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ProcessMounter(entities, entityListeners));
		mounters.computeIfAbsent("AssetAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.AssetMounter(entities, entityListeners));
		mounters.computeIfAbsent("PartAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.PartMounter(entities, entityListeners));
		mounters.computeIfAbsent("HardwareAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.HardwareMounter(entities, entityListeners));
		mounters.computeIfAbsent("ApplianceAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ApplianceMounter(entities, entityListeners));
		mounters.computeIfAbsent("ComputerAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ComputerMounter(entities, entityListeners));
		mounters.computeIfAbsent("SoftwareAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.SoftwareMounter(entities, entityListeners));
		mounters.computeIfAbsent("ObserverAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ObserverMounter(entities, entityListeners));
		mounters.computeIfAbsent("ServiceAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ServiceMounter(entities, entityListeners));
		mounters.computeIfAbsent("ApplicationAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.ApplicationMounter(entities, entityListeners));
		mounters.computeIfAbsent("ApplicationJavaAssertion", type -> new ArrayList<>(1)).add(new io.intino.cosmos.datahub.datamarts.master.mounters.JavaApplicationMounter(entities, entityListeners));
	}

	private static int availableThreads() {
		return Runtime.getRuntime().availableProcessors();
	}

	private javax.jms.Message requestResponseFromDatahub(String requestName, javax.jms.Message request) throws Exception {
    	long timeout = retryConfig.initialTimeoutAmount;
    	for(int i = 0; i < retryConfig.maxAttempts; i++) {
    		javax.jms.Message message = connector.requestResponse(DATAHUB_MESSAGE_TOPIC, request, timeout, retryConfig.timeoutUnit);
    		if(message != null) return message;
    		if(i < retryConfig.maxAttempts - 1) Logger.warn("("+(i+1)+") Datahub did not respond after " + timeout + " " + retryConfig.timeoutUnit + " to the request '" + requestName + "'. Trying again...");
    		timeout *= retryConfig.timeoutMultiplier;
    	}
    	throw new RuntimeException("Datahub did not respond to the request '" + requestName + "' after " + retryConfig.maxAttempts);
    }


	private class TimelineNodeImpl implements TimelineNode {

		private final String id;
		private final String type;
		private final Set<String> sources;
		private volatile File file;
		private volatile io.intino.sumus.chronos.TimelineFile timelineFile;
		private volatile java.lang.ref.SoftReference<io.intino.sumus.chronos.Timeline> cache;
		private volatile EventListener listener;
		private volatile boolean disposed;

		private TimelineNodeImpl(String id, String type, Set<String> sources) {
			this(id, type, sources, null);
		}

		private TimelineNodeImpl(String id, String type, Set<String> sources, File file) {
			this.id = requireNonNull(id);
			this.type = requireNonNull(type);
			this.sources = requireNonNull(sources);
			this.file = file;
		}

		@Override
		public String id() {
			return id;
		}

		@Override
		public String type() {
			return type;
		}


	    public boolean exists() {
	    	return timelineFile() != null;
	    }

		@Override
		public void dispose() {
			synchronized(this) {
				if (disposed) return;
				clearCache();
				listener = null;
				timelines.remove(id + ":" + type);
				disposed = true;
			}
		}

		@Override
		public io.intino.sumus.chronos.TimelineFile.TimeModel timeModel() {
			return timelineFile().timeModel();
		}

		@Override
		public io.intino.sumus.chronos.TimelineFile.SensorModel sensorModel() {
			return timelineFile().sensorModel();
		}

		@Override
		public Instant first() {
			return timelineFile().first();
		}

		@Override
		public Instant last() {
			return timelineFile().last();
		}

		@Override
		public io.intino.sumus.chronos.Timeline get() {
			synchronized(this) {
				if (disposed) throw new IllegalStateException("This " + getClass().getSimpleName() + " is disposed.");
				try {
					if (cache != null) {
						io.intino.sumus.chronos.Timeline timeline = cache.get();
						if (timeline != null) return timeline;
					}
					io.intino.sumus.chronos.Timeline timeline = timelineFile().timeline();
					cache = new java.lang.ref.SoftReference<>(timeline);
					return timeline;
				} catch(RuntimeException e) {
					throw e;
				} catch(Exception e) {
					throw new RuntimeException(e);
				}
			}
		}

		private io.intino.sumus.chronos.TimelineFile timelineFile() {
			synchronized(this) {
				if (disposed) throw new IllegalStateException("This " + getClass().getSimpleName() + " is disposed.");
				try {
					if (timelineFile != null) return timelineFile;
					return timelineFile = (!TimelineNode.AlwaysDownloadFromDatahub.get() && file != null && file.exists())
						? loadFile()
						: downloadFromDatahub();
				} catch(Exception e) {
					throw new RuntimeException(e);
				}
			}
		}

		@Override
		public void setEventListener(EventListener listener) {
			this.listener = listener;
		}

		private void notifyEvent(io.intino.alexandria.event.Event event) {
			synchronized(this) {
				if (disposed) return;
				try {
					if (!sources.contains(event.type())) return;
					clearCache();
					if (listener != null) listener.onEventReceived(this, event);
				} catch(Throwable e) {
					Logger.error(e);
				}
			}
		}

		private void clearCache() {
			if (cache != null) {
				cache.enqueue();
				cache = null;
			}
			timelineFile = null;
			file = null;
		}

		private io.intino.sumus.chronos.TimelineFile loadFile() throws Exception {
			return io.intino.sumus.chronos.TimelineFile.open(file);
		}

		private io.intino.sumus.chronos.TimelineFile downloadFromDatahub() throws Exception {
			javax.jms.Message response = requestResponseFromDatahub(
				"get-timeline=" + id(),
			 	request(TimelineNode.AlwaysDownloadFromDatahub.get() ? "download" : "path"));
			if (!response.getBooleanProperty("success")) return null;
			if (response instanceof javax.jms.TextMessage textResponse) {
				file = getFile(textResponse);
				if (file != null && file.exists()) return loadFile();
				file = null;
				response = requestResponseFromDatahub("get-timeline=" + id(), request("download"));
			}
			if (!response.getBooleanProperty("success")) return null;
			return readFromBytes((javax.jms.BytesMessage) response);
		}

		private io.intino.sumus.chronos.TimelineFile readFromBytes(javax.jms.BytesMessage m) throws Exception {
			int size = m.getIntProperty("size");
			byte[] bytes = new byte[size];
			m.readBytes(bytes, size);

			file = File.createTempFile(id(), ".timeline");
			java.nio.file.Files.write(file.toPath(), bytes, java.nio.file.StandardOpenOption.CREATE);
			file.deleteOnExit();

			return loadFile();
		}

		private File getFile(javax.jms.TextMessage m) {
			try {
				return new File(m.getText());
			} catch(Exception e) {
				return null;
			}
		}

		private javax.jms.Message request(String mode) throws Exception {
			ActiveMQTextMessage message = new ActiveMQTextMessage();
			String command = "datamart=" + name() + ";operation=get-timeline;id=" + id() + ";mode=" + mode + ";type=" + type;
			message.setText(command);
			return message;
		}
	}


	private class ReelNodeImpl implements ReelNode {
		private final String id;
		private final String type;
		private final Set<String> sources;
		private volatile File file;
		private volatile io.intino.sumus.chronos.ReelFile reelFile;
		private volatile EventListener listener;
		private volatile boolean disposed;

		private ReelNodeImpl(String id, String type, Set<String> sources) {
			this(id, type, sources, null);
		}

		private ReelNodeImpl(String id, String type, Set<String> sources, File file) {
			this.id = requireNonNull(id);
			this.type = requireNonNull(type);
			this.sources = requireNonNull(sources);
			this.file = file;
		}

		@Override
		public String id() {
			return id;
		}

		@Override
		public String type() {
			return type;
		}

		@Override
		public void dispose() {
			synchronized(this) {
				if (disposed) return;
				clearCache();
				listener = null;
				reels.remove(id + ":" + type);
				disposed = true;
			}
		}

		@Override
		public io.intino.sumus.chronos.ReelFile.Group groupOf(String signal) {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			if (reelFile == null) return null;
			return reelFile.groupOf(signal);
		}

		@Override
		public io.intino.sumus.chronos.Reel.State stateOf(String signal) {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			if (reelFile == null) throw new IllegalArgumentException("Reel file not found");
			return reelFile.stateOf(signal);
		}

		@Override
		public List<io.intino.sumus.chronos.Reel.State> stateOf(Stream<String> signals) {
			io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			if (reelFile == null) throw new IllegalArgumentException("Reel file not found");
			return signals.map(reelFile::stateOf).toList();
		}

		@Override
	    public io.intino.sumus.chronos.Reel get(io.intino.sumus.chronos.Period period) {
	    	io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			if (reelFile == null) throw new IllegalArgumentException("Reel file not found");
	    	return reelFile().reel().by(period);
	    }

		@Override
	    public io.intino.sumus.chronos.Reel get(Instant from, Instant to, io.intino.sumus.chronos.Period period) {
	    	io.intino.sumus.chronos.ReelFile reelFile = reelFile();
			if (reelFile == null) throw new IllegalArgumentException("Reel file not found");
	    	return reelFile.reel(from, to).by(period);
	    }

	    public boolean exists() {
	    	return reelFile() != null;
	    }

		private io.intino.sumus.chronos.ReelFile reelFile() {
			synchronized(this) {
				if (disposed) throw new IllegalStateException("This " + getClass().getSimpleName() + " is disposed.");
				try {
					if (reelFile != null) return reelFile;
					return reelFile = (file != null && file.exists())
						? loadFile()
						: downloadFromDatahub();
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			}
		}

		@Override
		public void setEventListener(EventListener listener) {
			this.listener = listener;
		}

		private void notifyEvent(io.intino.alexandria.event.Event event) {
			synchronized(this) {
				if (disposed) return;
				try {
					if (!sources.contains(event.type())) return;
					clearCache();
					if (listener != null) listener.onEventReceived(this, event);
				} catch(Throwable e) {
					Logger.error(e);
				}
			}
		}

		private void clearCache() {
			reelFile = null;
			file = null;
		}

		private io.intino.sumus.chronos.ReelFile loadFile() throws Exception {
			return io.intino.sumus.chronos.ReelFile.open(file);
		}

		private io.intino.sumus.chronos.ReelFile downloadFromDatahub() throws Exception {
			javax.jms.Message response = requestResponseFromDatahub("get-reel=" + id(), request("path"));
			if (!response.getBooleanProperty("success")) return null;
			if (response instanceof javax.jms.TextMessage textResponse) {
				file = getFile(textResponse);
				if (file != null && file.exists()) return loadFile();
				file = null;
				response = requestResponseFromDatahub("get-reel=" + id(), request("download"));
			}
			if (!response.getBooleanProperty("success")) return null;
			return readFromBytes((javax.jms.BytesMessage) response);
		}

		private io.intino.sumus.chronos.ReelFile readFromBytes(javax.jms.BytesMessage m) throws Exception {
			int size = m.getIntProperty("size");
			byte[] bytes = new byte[size];
			m.readBytes(bytes, size);
			file = File.createTempFile(id(), ".reel");
			java.nio.file.Files.write(file.toPath(), bytes, java.nio.file.StandardOpenOption.CREATE);
			file.deleteOnExit();
			return loadFile();
		}

		private File getFile(javax.jms.TextMessage m) {
			try {
				return new File(m.getText());
			} catch(Exception e) {
				return null;
			}
		}

		private javax.jms.Message request(String mode) throws Exception {
			ActiveMQTextMessage message = new ActiveMQTextMessage();
			String command = "datamart=" + name() + ";operation=get-reel;id=" + id() + ";mode=" + mode + ";type=" + type;
			message.setText(command);
			return message;
		}
	}

	private static Set<String> sourcesOfTimeline(String type) {
    	return switch(type) {
    		case "Computer" -> Set.of("Computer","ComputerAssertion");
    		case "Switch" -> Set.of("Switch","ApplianceAssertion");
    		case "CellularGateway" -> Set.of("CellularGateway","ApplianceAssertion");
    		case "Wireless" -> Set.of("Wireless","ApplianceAssertion");
    		case "Camera" -> Set.of("Camera","ApplianceAssertion");
    		case "Application" -> Set.of("Application","ApplicationAssertion");
    		default -> java.util.Collections.emptySet();
    	};
    }

	private static Set<String> sourcesOfReel(String type) {
    	return switch(type) {
    		case "Status" -> Set.of("Datalake#dl$status");
    		default -> java.util.Collections.emptySet();
    	};
    }

	// WARNING: extremely compacted and ugly code ahead... continue at your own discretion.
	public static final class MasterDatamartDefinition implements DatamartDefinition {

		@Override
		public String name() {
			return "master";
		}

		@Override
		public Scale scale() {
			return Scale.Day;
		}

		@Override
		public Query<EntityDefinition> entities() {
			return new Query<EntityDefinition>(List.of(placeEntityDefinition,areaEntityDefinition,incidentRuleEntityDefinition,modelEntityDefinition,orderTypeEntityDefinition,observableEntityDefinition,personEntityDefinition,processEntityDefinition,assetEntityDefinition,partEntityDefinition,hardwareEntityDefinition,applianceEntityDefinition,computerEntityDefinition,softwareEntityDefinition,observerEntityDefinition,serviceEntityDefinition,applicationEntityDefinition,javaApplicationEntityDefinition));
		}

		@Override
		public Query<StructDefinition> structs() {
			return new Query<StructDefinition>(List.of(model$ProfileStructDefinition,model$Profile$VariableStructDefinition,observable$OperationStructDefinition,observable$Operation$ProcedureStructDefinition,observable$Operation$Procedure$ParameterStructDefinition));
		}

		@Override
		public Optional<EntityDefinition> entity(String name) {
			switch(name) {
				case "Place": return Optional.of(placeEntityDefinition);
				case "Area": return Optional.of(areaEntityDefinition);
				case "IncidentRule": return Optional.of(incidentRuleEntityDefinition);
				case "Model": return Optional.of(modelEntityDefinition);
				case "OrderType": return Optional.of(orderTypeEntityDefinition);
				case "Observable": return Optional.of(observableEntityDefinition);
				case "Person": return Optional.of(personEntityDefinition);
				case "Process": return Optional.of(processEntityDefinition);
				case "Asset": return Optional.of(assetEntityDefinition);
				case "Part": return Optional.of(partEntityDefinition);
				case "Hardware": return Optional.of(hardwareEntityDefinition);
				case "Appliance": return Optional.of(applianceEntityDefinition);
				case "Computer": return Optional.of(computerEntityDefinition);
				case "Software": return Optional.of(softwareEntityDefinition);
				case "Observer": return Optional.of(observerEntityDefinition);
				case "Service": return Optional.of(serviceEntityDefinition);
				case "Application": return Optional.of(applicationEntityDefinition);
				case "JavaApplication": return Optional.of(javaApplicationEntityDefinition);
			}
			return Optional.empty();
		}

		@Override
    	public Optional<StructDefinition> struct(String name) {
    		switch(name) {
    			case "Model$Profile": return Optional.of(model$ProfileStructDefinition);
    			case "Model$Profile$Variable": return Optional.of(model$Profile$VariableStructDefinition);
    			case "Observable$Operation": return Optional.of(observable$OperationStructDefinition);
    			case "Observable$Operation$Procedure": return Optional.of(observable$Operation$ProcedureStructDefinition);
    			case "Observable$Operation$Procedure$Parameter": return Optional.of(observable$Operation$Procedure$ParameterStructDefinition);
    		}
    		return Optional.empty();
    	}

		private MasterDatamartDefinition datamart() {
			return this;
		}

		public final EntityDefinition placeEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Place";}
			public String name() {return "Place";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Place.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "label";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "coordinates";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "timeOffset";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "region";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "city";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "area";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Area.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "address";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "postalCode";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition areaEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Area";}
			public String name() {return "Area";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Area.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "label";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition incidentRuleEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "IncidentRule";}
			public String name() {return "IncidentRule";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "description";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "target";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "model";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "level";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.Level.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "tracking";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "triggers";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "precondition";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "condition";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "countermeasures";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "orderTypes";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition modelEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Model";}
			public String name() {return "Model";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "type";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "target";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "profile";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition orderTypeEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "OrderType";}
			public String name() {return "OrderType";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.OrderType.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition observableEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable";}
			public String name() {return "Observable";}
			public boolean isAbstract() {return true;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.empty();}
			public List<EntityDefinition> ancestors() {return java.util.List.of();}
			public List<EntityDefinition> descendants() {return java.util.List.of(personEntityDefinition,processEntityDefinition,assetEntityDefinition,partEntityDefinition,hardwareEntityDefinition,applianceEntityDefinition,computerEntityDefinition,softwareEntityDefinition,observerEntityDefinition,serviceEntityDefinition,applicationEntityDefinition,javaApplicationEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
				list.add(new AttributeDefinition() {
					public String name() {return "id";}
					public Class<?> type() {return String.class;}
					public String toString() {return name();}
				});
				list.add(new AttributeDefinition() {
					public String name() {return "enabled";}
					public Class<?> type() {return Boolean.class;}
					public String toString() {return name();}
				});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "label";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "place";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Place.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "team";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "model";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "container";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "observer";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Observer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "responsible";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Person.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "tags";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "operationList";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().observable$OperationStructDefinition);}
		    				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition personEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Person";}
			public String name() {return "Person";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Person.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "area";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Area.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "role";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "supervisor";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Person.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "email";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "phone";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition processEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Process";}
			public String name() {return "Process";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Process.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition assetEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Asset";}
			public String name() {return "Asset";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Asset.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition partEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Part";}
			public String name() {return "Part";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Part.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition hardwareEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Hardware";}
			public String name() {return "Hardware";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of(applianceEntityDefinition,computerEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Hardware.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "sn";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "ip";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "publicIp";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition applianceEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Hardware.Appliance";}
			public String name() {return "Appliance";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(hardwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(hardwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Appliance.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition computerEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Hardware.Computer";}
			public String name() {return "Computer";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(hardwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(hardwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Computer.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "architecture";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "os";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "cores";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "diskSize";}
		    		public Class<?> type() {return Long.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "memorySize";}
		    		public Class<?> type() {return Long.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "processes";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "maxOpenFiles";}
		    		public Class<?> type() {return Long.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jvm";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "isp";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition softwareEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software";}
			public String name() {return "Software";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(observableEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of(observerEntityDefinition,serviceEntityDefinition,applicationEntityDefinition,javaApplicationEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Software.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition observerEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Observer";}
			public String name() {return "Observer";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(softwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observer.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "version";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "installedActivities";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "enabledActivities";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition serviceEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Service";}
			public String name() {return "Service";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(softwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Service.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "url";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition applicationEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Application";}
			public String name() {return "Application";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(softwareEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of(javaApplicationEntityDefinition);}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Application.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "name";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "user";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "systemService";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "startingTime";}
		    		public Class<?> type() {return LocalDateTime.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "pid";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "commandLine";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "state";}
		    		public Class<?> type() {return io.intino.cosmos.datahub.datamarts.master.entities.Application.State.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};
		public final EntityDefinition javaApplicationEntityDefinition = new EntityDefinition() {
			private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
			public String fullName() {return "Observable.Software.Application.JavaApplication";}
			public String name() {return "JavaApplication";}
			public boolean isAbstract() {return false;}
			public List<AttributeDefinition> declaredAttributes() {	return declaredAttributes;}
			public Optional<EntityDefinition> parent() {return Optional.of(applicationEntityDefinition);}
			public List<EntityDefinition> ancestors() {return java.util.List.of(applicationEntityDefinition,softwareEntityDefinition,observableEntityDefinition);}
			public List<EntityDefinition> descendants() {return java.util.List.of();}
			public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.JavaApplication.class;}
			private List<AttributeDefinition> initAttributeDefinitions() {
				List<AttributeDefinition> list = new ArrayList<>();
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "classpathPrefix";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "mainArtifact";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "debugPort";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jmxPort";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "minMemory";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "maxMemory";}
		    		public Class<?> type() {return Integer.class;}
		    		public String toString() {return name();}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "classpath";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "inputArguments";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jvmParameter";}
		    		public Class<?> type() {return java.util.List.class;}
		    		public String toString() {return name();}
		    		public List<ParameterDefinition> parameters() {
		    			return List.of(new ParameterDefinition() {
		    				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
		    				public Class<?> javaClass() {return String.class;}
		    				public String toString() {return javaClass().getSimpleName();}
		    			});
		    		}
		    	});
		    	list.add(new AttributeDefinition() {
		    		public String name() {return "jvmVersion";}
		    		public Class<?> type() {return String.class;}
		    		public String toString() {return name();}
		    	});
		    	return Collections.synchronizedList(list);
			}
			public boolean equals(Object other) {
				if(other == null || other.getClass() != getClass()) return false;
				return fullName().equals(((EntityDefinition)other).fullName());
			}
			public int hashCode() {return fullName().hashCode();}
			public String toString() {return fullName();}
		};

        public final StructDefinition model$ProfileStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Model$Profile";}
        	public String name() {return "Profile";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "variableList";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().model$Profile$VariableStructDefinition);}
            				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.Variable.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if(other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition model$Profile$VariableStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Model$Profile$Variable";}
        	public String name() {return "Variable";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Model.Profile.Variable.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "className";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "label";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "type";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "operator";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "unit";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "symbol";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "values";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.empty();}
            				public Class<?> javaClass() {return String.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "min";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "max";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "description";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "format";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if(other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition observable$OperationStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Observable$Operation";}
        	public String name() {return "Operation";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "activity";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "procedureList";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().observable$Operation$ProcedureStructDefinition);}
            				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if(other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition observable$Operation$ProcedureStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Observable$Operation$Procedure";}
        	public String name() {return "Procedure";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "returnType";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "description";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "parameterList";}
            		public Class<?> type() {return java.util.List.class;}
            		public String toString() {return name();}
            		public List<ParameterDefinition> parameters() {
            			return List.of(new ParameterDefinition() {
            				public Optional<ConceptDefinition<?>> asConceptDefinition() {return Optional.of(datamart().observable$Operation$Procedure$ParameterStructDefinition);}
            				public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.Parameter.class;}
            				public String toString() {return javaClass().getSimpleName();}
            			});
            		}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if(other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
        public final StructDefinition observable$Operation$Procedure$ParameterStructDefinition = new StructDefinition() {
        	private final List<AttributeDefinition> declaredAttributes = initAttributeDefinitions();
        	public String fullName() {return "Observable$Operation$Procedure$Parameter";}
        	public String name() {return "Parameter";}
        	public List<AttributeDefinition> declaredAttributes() {return declaredAttributes;}
        	public Optional<StructDefinition> parent() {return Optional.empty();}
        	public List<StructDefinition> ancestors() {return java.util.List.of();}
        	public List<StructDefinition> descendants() {return java.util.List.of();}
        	public Class<?> javaClass() {return io.intino.cosmos.datahub.datamarts.master.entities.Observable.Operation.Procedure.Parameter.class;}
        	private List<AttributeDefinition> initAttributeDefinitions() {
        		List<AttributeDefinition> list = new ArrayList<>();
            	list.add(new AttributeDefinition() {
            		public String name() {return "type";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	list.add(new AttributeDefinition() {
            		public String name() {return "name";}
            		public Class<?> type() {return String.class;}
            		public String toString() {return name();}
            	});
            	return Collections.synchronizedList(list);
        	}
        	public boolean equals(Object other) {
            	if(other == null || other.getClass() != getClass()) return false;
            	return fullName().equals(((StructDefinition)other).fullName());
            }
            public int hashCode() {return fullName().hashCode();}
            public String toString() {return fullName();}
        };
	}

	static {
		try {
			Object ref = MasterDatamart.class.getDeclaredField("definition").get(null);
			java.lang.reflect.Field field = ref.getClass().getDeclaredField("definition");
			field.setAccessible(true);
			field.set(ref, definition);
			field.setAccessible(false);
		} catch (Exception e) {
			throw new ExceptionInInitializerError("Could not set MasterDatamart.definition field");
		}
	}
}