package io.intino.cosmos.datahub.datamarts.master.mounters;

import io.intino.ness.master.Datamart.EntityListener;
import io.intino.cosmos.datahub.datamarts.master.MasterDatamart;

import java.time.*;
import java.util.*;
import java.util.stream.*;

import io.intino.alexandria.event.Event;
import io.intino.alexandria.event.message.MessageEvent;
import io.intino.alexandria.message.Message;
import io.intino.cosmos.datahub.datamarts.master.MasterEntity;
import io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule;

import static io.intino.cosmos.datahub.datamarts.master.MasterMounter.Operation.*;

public class IncidentRuleMounter implements io.intino.cosmos.datahub.datamarts.master.MasterMounter {

	private final MasterDatamart.Entities entities;
	private final List<EntityListener> listeners;
	private boolean useListeners = true;

	public IncidentRuleMounter(MasterDatamart.Entities entities, List<EntityListener> listeners) {
		this.entities = entities;
		this.listeners = listeners;
	}

	@Override
	public IncidentRuleMounter useListeners(boolean useListeners) {
		this.useListeners = useListeners;
		return this;
	}

	@Override
	public void mount(Event event) {
		Message message = ((MessageEvent)event).toMessage();
		String id = message.get("id").asString();
		Operation[] operation = {Update};
		MasterEntity entity = findOrCreateEntity(id, operation);
		updateAttributes(message, operation, entity);
		if(operation[0] != Skip && useListeners) notifyListeners(operation[0], entity);
	}

	private void updateAttributes(Message message, Operation[] operation, MasterEntity owner) {
		for(String attr : message.attributes()) {
			update(owner, attr, parse(attr, message));
			if(attr.equals("enabled")) {
				if(message.get("enabled").asBoolean()) {
					entities.enable(owner.id());
				} else {
					entities.disable(owner.id());
				}
			}
		}
	}

	private MasterEntity findOrCreateEntity(String id, Operation[] operation) {
		MasterEntity entity = entities.get(io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.definition, id);
		if(entity == null) entity = entities.getDisabled(io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.definition, id);
		if(entity == null) {
			entity = new IncidentRule(id, entities.datamart());
			entities.add(entity);
			operation[0] = Create;
		}
		return entity;
	}

	private void notifyListeners(Operation operation, MasterEntity entity) {
		for(EntityListener listener : listeners) {
			switch(operation) {
				case Create: new Thread(() -> listener.onCreate(entity), "MasterListener-IncidentRuleMounter-" + System.currentTimeMillis()).start(); break;
				case Update: new Thread(() -> listener.onUpdate(entity), "MasterListener-IncidentRuleMounter-" + System.currentTimeMillis()).start(); break;
				case Remove: new Thread(() -> listener.onRemove(entity), "MasterListener-IncidentRuleMounter-" + System.currentTimeMillis()).start(); break;
			}
		}
	}

	private Object parse(String attribute, Message message) {
		if(message.get(attribute).isNull()) return null;
		switch(attribute) {
			case "target": return parseTarget(message);
			case "model": return parseModel(message);
			case "condition": return parseCondition(message);
			case "precondition": return parsePrecondition(message);
			case "level": return parseLevel(message);
			case "tracking": return parseTracking(message);
			case "dismissCondition": return parseDismissCondition(message);
			case "description": return parseDescription(message);
			case "triggers": return parseTriggers(message);
		}
		return message.get(attribute).asString();
	}

	private String parseTarget(Message m) {
		return m.get("target").as(String.class);
	}

	private String parseModel(Message m) {
		return m.get("model").as(String.class);
	}

	private String parseCondition(Message m) {
		return m.get("condition").as(String.class);
	}

	private String parsePrecondition(Message m) {
		return m.get("precondition").as(String.class);
	}

	private io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.Level parseLevel(Message m) {
		try {
			return io.intino.cosmos.datahub.datamarts.master.entities.IncidentRule.Level.valueOf(m.get("level").asString());
		} catch(Exception ignored) {
			return null;
		}
	}

	private String parseTracking(Message m) {
		return m.get("tracking").as(String.class);
	}

	private String parseDismissCondition(Message m) {
		return m.get("dismissCondition").as(String.class);
	}

	private String parseDescription(Message m) {
		return m.get("description").as(String.class);
	}

	private java.util.List<String> parseTriggers(Message m) {
		var value = m.get("triggers");
		return value.data().isEmpty() ? java.util.Collections.emptyList() : value.asList(String.class);
	}
}