/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.internal.storage.commitgraph;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.Stack;
import org.eclipse.jgit.annotations.NonNull;
import org.eclipse.jgit.annotations.Nullable;
import org.eclipse.jgit.errors.CorruptObjectException;
import org.eclipse.jgit.errors.IncorrectObjectTypeException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.commitgraph.ChangedPathFilter;
import org.eclipse.jgit.internal.storage.commitgraph.GraphCommits;
import org.eclipse.jgit.internal.storage.io.CancellableDigestOutputStream;
import org.eclipse.jgit.lib.ObjectReader;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.treewalk.EmptyTreeIterator;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.eclipse.jgit.util.NB;

public class CommitGraphWriter {
    private static final int COMMIT_GRAPH_VERSION_GENERATED = 1;
    private static final int OID_HASH_VERSION = 1;
    private static final int GRAPH_FANOUT_SIZE = 1024;
    private static final int GENERATION_NUMBER_MAX = 0x3FFFFFFF;
    private static final int MAX_CHANGED_PATHS = 512;
    private final int hashsz;
    private final GraphCommits graphCommits;
    private final boolean generateChangedPathFilters;

    public CommitGraphWriter(@NonNull GraphCommits graphCommits) {
        this(graphCommits, false);
    }

    public CommitGraphWriter(@NonNull GraphCommits graphCommits, boolean generateChangedPathFilters) {
        this.graphCommits = graphCommits;
        this.hashsz = 20;
        this.generateChangedPathFilters = generateChangedPathFilters;
    }

    public Stats write(@NonNull ProgressMonitor monitor, @NonNull OutputStream commitGraphStream) throws IOException {
        if (this.graphCommits.size() == 0) {
            return Stats.EMPTY;
        }
        BloomFilterChunks bloomFilterChunks = this.generateChangedPathFilters ? this.computeBloomFilterChunks(monitor) : null;
        ArrayList<ChunkHeader> chunks = new ArrayList<ChunkHeader>();
        chunks.addAll(CommitGraphWriter.createCoreChunks(this.hashsz, this.graphCommits));
        chunks.addAll(CommitGraphWriter.createBloomFilterChunkHeaders(bloomFilterChunks));
        chunks = Collections.unmodifiableList(chunks);
        long expectedSize = CommitGraphWriter.calculateExpectedSize(chunks);
        try {
            Throwable throwable = null;
            Object var8_9 = null;
            try (CancellableDigestOutputStream out = new CancellableDigestOutputStream(monitor, commitGraphStream);){
                this.writeHeader(out, chunks.size());
                this.writeChunkLookup(out, chunks);
                this.writeChunks(out, chunks);
                this.writeCheckSum(out);
                if (expectedSize != out.length()) {
                    throw new IllegalStateException(String.format(JGitText.get().commitGraphUnexpectedSize, expectedSize, out.length()));
                }
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (InterruptedIOException e) {
            throw new IOException(JGitText.get().commitGraphWritingCancelled, e);
        }
        return Stats.from(bloomFilterChunks);
    }

    private static List<ChunkHeader> createCoreChunks(int hashsz, GraphCommits graphCommits) {
        ArrayList<ChunkHeader> chunks = new ArrayList<ChunkHeader>();
        chunks.add(new ChunkHeader(1330201670, 1024L));
        chunks.add(new ChunkHeader(1330201676, hashsz * graphCommits.size()));
        chunks.add(new ChunkHeader(1128546644, (hashsz + 16) * graphCommits.size()));
        if (graphCommits.getExtraEdgeCnt() > 0) {
            chunks.add(new ChunkHeader(0x45444745, graphCommits.getExtraEdgeCnt() * 4));
        }
        return Collections.unmodifiableList(chunks);
    }

    private static List<ChunkHeader> createBloomFilterChunkHeaders(@Nullable BloomFilterChunks bloomFilterChunks) {
        ArrayList<ChunkHeader> chunks = new ArrayList<ChunkHeader>();
        if (bloomFilterChunks != null) {
            chunks.add(new ChunkHeader(1112097880, bloomFilterChunks.index));
            chunks.add(new ChunkHeader(1111769428, bloomFilterChunks.data));
        }
        return Collections.unmodifiableList(chunks);
    }

    private static long calculateExpectedSize(List<ChunkHeader> chunks) {
        int chunkLookup = (chunks.size() + 1) * 12;
        long chunkContent = chunks.stream().mapToLong(c -> c.size).sum();
        return (long)(8 + chunkLookup) + chunkContent + 20L;
    }

    private void writeHeader(CancellableDigestOutputStream out, int numChunks) throws IOException {
        byte[] headerBuffer = new byte[8];
        NB.encodeInt32(headerBuffer, 0, 1128747080);
        byte[] byArray = new byte[4];
        byArray[0] = 1;
        byArray[1] = 1;
        byArray[2] = (byte)numChunks;
        byte[] buff = byArray;
        System.arraycopy(buff, 0, headerBuffer, 4, 4);
        out.write(headerBuffer, 0, 8);
        out.flush();
    }

    private void writeChunkLookup(CancellableDigestOutputStream out, List<ChunkHeader> chunks) throws IOException {
        int numChunks = chunks.size();
        long chunkOffset = 8L + ((long)numChunks + 1L) * 12L;
        byte[] buffer = new byte[12];
        for (ChunkHeader chunk : chunks) {
            NB.encodeInt32(buffer, 0, chunk.id);
            NB.encodeInt64(buffer, 4, chunkOffset);
            out.write(buffer);
            chunkOffset += chunk.size;
        }
        NB.encodeInt32(buffer, 0, 0);
        NB.encodeInt64(buffer, 4, chunkOffset);
        out.write(buffer);
    }

    private void writeChunks(CancellableDigestOutputStream out, List<ChunkHeader> chunks) throws IOException {
        for (ChunkHeader chunk : chunks) {
            int chunkId = chunk.id;
            switch (chunkId) {
                case 1330201670: {
                    this.writeFanoutTable(out);
                    break;
                }
                case 1330201676: {
                    this.writeOidLookUp(out);
                    break;
                }
                case 1128546644: {
                    this.writeCommitData(out);
                    break;
                }
                case 0x45444745: {
                    this.writeExtraEdges(out);
                    break;
                }
                case 1111769428: 
                case 1112097880: {
                    if (!chunk.data.isPresent()) {
                        throw new IllegalStateException("data for this chunk must be precomputed");
                    }
                    chunk.data.get().writeTo(out);
                    break;
                }
                default: {
                    throw new IllegalStateException("Don't know how to write chunk " + chunkId);
                }
            }
        }
    }

    private void writeCheckSum(CancellableDigestOutputStream out) throws IOException {
        out.write(out.getDigest());
        out.flush();
    }

    private void writeFanoutTable(CancellableDigestOutputStream out) throws IOException {
        byte[] tmp = new byte[4];
        int[] fanout = new int[256];
        for (RevCommit c : this.graphCommits) {
            int n = c.getFirstByte() & 0xFF;
            fanout[n] = fanout[n] + 1;
        }
        int i = 1;
        while (i < fanout.length) {
            int n = i;
            fanout[n] = fanout[n] + fanout[i - 1];
            ++i;
        }
        int[] nArray = fanout;
        int n = fanout.length;
        int n2 = 0;
        while (n2 < n) {
            int n3 = nArray[n2];
            NB.encodeInt32(tmp, 0, n3);
            out.write(tmp, 0, 4);
            ++n2;
        }
    }

    private void writeOidLookUp(CancellableDigestOutputStream out) throws IOException {
        byte[] tmp = new byte[4 + this.hashsz];
        for (RevCommit c : this.graphCommits) {
            c.copyRawTo(tmp, 0);
            out.write(tmp, 0, this.hashsz);
        }
    }

    private void writeCommitData(CancellableDigestOutputStream out) throws IOException {
        ProgressMonitor monitor = out.getWriteMonitor();
        int[] generations = this.computeGenerationNumbers(monitor);
        monitor.beginTask(JGitText.get().writingOutCommitGraph, this.graphCommits.size());
        int num = 0;
        byte[] tmp = new byte[this.hashsz + 16];
        int i = 0;
        for (RevCommit commit : this.graphCommits) {
            RevCommit parent;
            int edgeValue;
            int[] packedDate = new int[2];
            RevTree treeId = commit.getTree();
            treeId.copyRawTo(tmp, 0);
            RevCommit[] parents = commit.getParents();
            if (parents.length == 0) {
                edgeValue = 0x70000000;
            } else {
                parent = parents[0];
                edgeValue = this.graphCommits.getOidPosition(parent);
            }
            NB.encodeInt32(tmp, this.hashsz, edgeValue);
            if (parents.length == 1) {
                edgeValue = 0x70000000;
            } else if (parents.length == 2) {
                parent = parents[1];
                edgeValue = this.graphCommits.getOidPosition(parent);
            } else if (parents.length > 2) {
                edgeValue = Integer.MIN_VALUE | num;
                num += parents.length - 1;
            }
            NB.encodeInt32(tmp, this.hashsz + 4, edgeValue);
            packedDate[0] = 0;
            packedDate[0] = packedDate[0] | generations[i] << 2;
            packedDate[1] = commit.getCommitTime();
            NB.encodeInt32(tmp, this.hashsz + 8, packedDate[0]);
            NB.encodeInt32(tmp, this.hashsz + 12, packedDate[1]);
            out.write(tmp);
            monitor.update(1);
            ++i;
        }
        monitor.endTask();
    }

    private int[] computeGenerationNumbers(ProgressMonitor monitor) throws MissingObjectException {
        int[] generations = new int[this.graphCommits.size()];
        monitor.beginTask(JGitText.get().computingCommitGeneration, this.graphCommits.size());
        for (RevCommit cmit : this.graphCommits) {
            monitor.update(1);
            int generation = generations[this.graphCommits.getOidPosition(cmit)];
            if (generation != 0 && generation != Integer.MAX_VALUE) continue;
            Stack<RevCommit> commitStack = new Stack<RevCommit>();
            commitStack.push(cmit);
            while (!commitStack.empty()) {
                int maxGeneration = 0;
                boolean allParentComputed = true;
                RevCommit current = (RevCommit)commitStack.peek();
                int i = 0;
                while (i < current.getParentCount()) {
                    RevCommit parent = current.getParent(i);
                    generation = generations[this.graphCommits.getOidPosition(parent)];
                    if (generation == 0 || generation == Integer.MAX_VALUE) {
                        allParentComputed = false;
                        commitStack.push(parent);
                        break;
                    }
                    if (generation > maxGeneration) {
                        maxGeneration = generation;
                    }
                    ++i;
                }
                if (!allParentComputed) continue;
                RevCommit commit = (RevCommit)commitStack.pop();
                generation = maxGeneration + 1;
                if (generation > 0x3FFFFFFF) {
                    generation = 0x3FFFFFFF;
                }
                generations[this.graphCommits.getOidPosition((RevCommit)commit)] = generation;
            }
        }
        monitor.endTask();
        return generations;
    }

    /*
     * Unable to fully structure code
     */
    private static Optional<HashSet<ByteBuffer>> computeBloomFilterPaths(ObjectReader or, RevCommit cmit) throws MissingObjectException, IncorrectObjectTypeException, CorruptObjectException, IOException {
        paths = new HashSet<ByteBuffer>();
        var3_3 = null;
        var4_5 = null;
        try {
            walk = new TreeWalk(null, or);
            try {
                walk.setRecursive(true);
                if (cmit.getParentCount() == 0) {
                    walk.addTree(new EmptyTreeIterator());
                } else {
                    walk.addTree(cmit.getParent(0).getTree());
                }
                walk.addTree(cmit.getTree());
                while (true) {
                    if (walk.idEqual(0, 1)) {
                    }
                    rawPath = walk.getRawPath();
                    paths.add(ByteBuffer.wrap(rawPath));
                    i = 0;
                    while (i < rawPath.length) {
                        if (rawPath[i] == 47) {
                            paths.add(ByteBuffer.wrap(rawPath, 0, i));
                        }
                        if (paths.size() > 512) {
                            return Optional.empty();
                        }
                        ++i;
                    }
                    break;
                }
            }
            finally {
                if (walk.next()) ** continue;
            }
        }
        catch (Throwable var4_6) {
            if (var3_3 == null) {
                var3_3 = var4_6;
            } else if (var3_3 != var4_6) {
                var3_3.addSuppressed(var4_6);
            }
            throw var3_3;
        }
        return Optional.of(paths);
    }

    private BloomFilterChunks computeBloomFilterChunks(ProgressMonitor monitor) throws MissingObjectException, IncorrectObjectTypeException, CorruptObjectException, IOException {
        ByteArrayOutputStream index = new ByteArrayOutputStream();
        ByteArrayOutputStream data = new ByteArrayOutputStream();
        long filtersReused = 0L;
        long filtersComputed = 0L;
        byte[] scratch = new byte[4];
        NB.encodeInt32(scratch, 0, 1);
        data.write(scratch);
        NB.encodeInt32(scratch, 0, 7);
        data.write(scratch);
        NB.encodeInt32(scratch, 0, 10);
        data.write(scratch);
        int dataHeaderSize = data.size();
        Throwable throwable = null;
        Object var11_10 = null;
        try (RevWalk rw = new RevWalk(this.graphCommits.getObjectReader());){
            monitor.beginTask(JGitText.get().computingPathBloomFilters, this.graphCommits.size());
            for (RevCommit cmit : this.graphCommits) {
                ChangedPathFilter cpf = cmit.getChangedPathFilter(rw);
                if (cpf != null) {
                    ++filtersReused;
                } else {
                    ++filtersComputed;
                    Optional<HashSet<ByteBuffer>> paths = CommitGraphWriter.computeBloomFilterPaths(this.graphCommits.getObjectReader(), cmit);
                    cpf = paths.isEmpty() ? ChangedPathFilter.FULL : ChangedPathFilter.fromPaths((Set<ByteBuffer>)paths.get());
                }
                cpf.writeTo(data);
                NB.encodeInt32(scratch, 0, data.size() - dataHeaderSize);
                index.write(scratch);
                monitor.update(1);
            }
            monitor.endTask();
            return new BloomFilterChunks(index, data, filtersReused, filtersComputed);
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private void writeExtraEdges(CancellableDigestOutputStream out) throws IOException {
        byte[] tmp = new byte[4];
        for (RevCommit commit : this.graphCommits) {
            RevCommit[] parents = commit.getParents();
            if (parents.length <= 2) continue;
            int n = 1;
            while (n < parents.length) {
                RevCommit parent = parents[n];
                int edgeValue = this.graphCommits.getOidPosition(parent);
                if (n == parents.length - 1) {
                    edgeValue |= Integer.MIN_VALUE;
                }
                NB.encodeInt32(tmp, 0, edgeValue);
                out.write(tmp);
                ++n;
            }
        }
    }

    private static class BloomFilterChunks {
        final ByteArrayOutputStream index;
        final ByteArrayOutputStream data;
        final long filtersReused;
        final long filtersComputed;

        BloomFilterChunks(ByteArrayOutputStream index, ByteArrayOutputStream data, long filtersReused, long filtersComputed) {
            this.index = index;
            this.data = data;
            this.filtersReused = filtersReused;
            this.filtersComputed = filtersComputed;
        }
    }

    private static class ChunkHeader {
        final int id;
        final long size;
        final Optional<ByteArrayOutputStream> data;

        public ChunkHeader(int id, long size) {
            this.id = id;
            this.size = size;
            this.data = Optional.empty();
        }

        ChunkHeader(int id, ByteArrayOutputStream data) {
            this.id = id;
            this.size = data.size();
            this.data = Optional.of(data);
        }
    }

    public static class Stats {
        static final Stats EMPTY = new Stats();
        private long changedPathFiltersReused = 0L;
        private long changedPathFiltersComputed = 0L;

        static final Stats from(@Nullable BloomFilterChunks bloomFilterChunks) {
            Stats stats = new Stats();
            if (bloomFilterChunks != null) {
                stats.changedPathFiltersComputed = bloomFilterChunks.filtersComputed;
                stats.changedPathFiltersReused = bloomFilterChunks.filtersReused;
            }
            return stats;
        }

        private Stats() {
        }

        public long getChangedPathFiltersReused() {
            return this.changedPathFiltersReused;
        }

        public long getChangedPathFiltersComputed() {
            return this.changedPathFiltersComputed;
        }
    }
}

