package io.intino.consul.container.box.os.local;

import com.google.gson.JsonObject;
import io.intino.alexandria.Json;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;
import io.intino.consul.framework.utils.Utils;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import oshi.SystemInfo;

import java.io.File;
import java.io.IOException;
import java.net.*;
import java.time.LocalDate;
import java.util.AbstractMap;
import java.util.Collections;
import java.util.Map;

import static java.time.LocalDate.now;

public class LocalSystem implements Activity.System {
	public static final String LOOP_BACK_ADDRESS = "127.0.0.1";
	private final SystemInfo si;
	private final String ispProviderToken;
	private Map.Entry<LocalDate, JsonObject> ipInfo;

	public LocalSystem(String ispProviderToken) {
		this.ispProviderToken = ispProviderToken;
		this.si = new SystemInfo();
		LogManager.getCurrentLoggers().asIterator().forEachRemaining(c -> ((org.apache.log4j.Logger) c).setLevel(Level.ERROR));
	}

	@Override
	public String name() {
		return Utils.calculateHostName();
	}

	@Override
	public OperatingSystem operatingSystem() {
		return new LocalOperatingSystem();
	}

	@Override
	public HDD hdd() {
		File file = new File(".");
		return new HDD(inMb(file.getTotalSpace()),
				inMb(file.getTotalSpace() - file.getFreeSpace()),
				si.getOperatingSystem().getCurrentProcess().getHardOpenFileLimit());
	}

	@Override
	public RAM ram() {
		return new RAM(inMb(si.getHardware().getMemory().getTotal()), 0);
	}

	@Override
	public CPU cpu() {
		return new CPU(si.getHardware().getProcessor().getLogicalProcessorCount(), 0, si.getHardware().getSensors().getCpuTemperature());
	}

	@Override
	public Network network() {
		return new Network(localIp(), publicIp(), isp());
	}

	private static long inMb(long value) {
		return value / (1024 * 1024);
	}

	private String localIp() {
		try {
			String hostAddress = Inet4Address.getLocalHost().getHostAddress();
			return hostAddress.equals(LOOP_BACK_ADDRESS) ? findInNetworkInterfaces() : hostAddress;
		} catch (UnknownHostException e) {
			return "0.0.0.0";
		}
	}

	private String publicIp() {
		if (ipInfo != null && now().equals(ipInfo.getKey())) return ipInfo.getValue().get("ip").getAsString();
		updateIpInfo();
		return ipInfo == null ? null : ipInfo.getValue().get("ip").getAsString();
	}

	private String isp() {
		if (ipInfo != null && now().equals(ipInfo.getKey())) return ipInfo.getValue().get("org").getAsString();
		updateIpInfo();
		return ipInfo == null ? null : ipInfo.getValue().get("org").getAsString();
	}

	private void updateIpInfo() {
		JsonObject jsonObject = ipInfoRequest();
		if (jsonObject != null) ipInfo = new AbstractMap.SimpleEntry<>(now(), jsonObject);
	}

	private JsonObject ipInfoRequest() {
		try {
			URL url = new URL("http://ipinfo.io");
			HttpURLConnection con = (HttpURLConnection) url.openConnection();
			if (ispProviderToken != null) con.setRequestProperty("token", ispProviderToken);
			con.setRequestProperty("Content-Type", "application/json");
			con.setConnectTimeout(5000);
			con.setReadTimeout(5000);
			con.setRequestMethod("GET");
			if (con.getResponseCode() > 210) return null;
			return Json.fromJson(new String(con.getInputStream().readAllBytes()).trim(), JsonObject.class);
		} catch (IOException e) {
			return null;
		}
	}

	private String findInNetworkInterfaces() {
		try {
			for (InetAddress address : Collections.list(NetworkInterface.getNetworkInterfaces().nextElement().getInetAddresses()))
				if (address instanceof Inet4Address && !address.isLoopbackAddress() && !LOOP_BACK_ADDRESS.equals(address.getHostAddress()))
					return address.getHostAddress();
		} catch (SocketException e) {
			Logger.error(e.getMessage());
		}
		return "0.0.0.0";
	}
}