package io.intino.consul.container.box.actions;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.ContainerBox;
import io.intino.consul.container.model.Info;
import io.intino.consul.framework.Activity;
import io.intino.cosmos.datahub.messages.monitoring.Status;
import io.intino.cosmos.datahub.messages.universe.ComputerAssertion;

import java.io.BufferedReader;
import java.io.IOException;
import java.time.Instant;
import java.util.List;
import java.util.Objects;

public class ComputerAssertionAction {
	public static final String ONLINE_SIGNAL = "online";
	public ContainerBox box;
	private Activity.Context context;

	public void execute() {
		try {
			doExecute();
		} catch (Throwable e) {
			Logger.error(e);
		}
	}

	private void doExecute() {
		context = box.context();
		final ComputerAssertion newInfo = computerInfo();
		if (newInfo == null) return;
		final Info last = last();
		if (!hasChanges(last, newInfo)) return;
		if (context.terminal() != null) {
			sleep(1000);
			context.terminal().publish(newInfo);
		}
		if (last == null && context.terminal() != null) context.terminal().publish(status(List.of(ONLINE_SIGNAL)));
		box.store().put("info", map(newInfo));
	}

	private Status status(List<String> signals) {
		return new Status(context.observer())
				.observable(context.hostName())
				.group(ONLINE_SIGNAL)
				.signals(signals);
	}

	private boolean hasChanges(Info last, ComputerAssertion newInfo) {
		return last == null
			   || !Objects.equals(last.name(), newInfo.id())
			   || last.diskSize() != newInfo.diskSize()
			   || !Objects.equals(last.os(), newInfo.os())
			   || last.memorySize() != newInfo.memorySize()
			   || last.maxOpenFiles() != newInfo.maxOpenFiles()
			   || !last.localIps().equals(newInfo.ip())
			   || !Objects.equals(last.publicIp(), newInfo.publicIp())
			   || !Objects.equals(newInfo.jvm(), last.jvm());
	}

	private Info last() {
		return box.store().get("info", Info.class);
	}

	private void sleep(int i) {
		try {
			Thread.sleep(i);
		} catch (InterruptedException e) {
			Logger.error(e);
		}
	}

	private Info map(ComputerAssertion newInfo) {
		return new Info(newInfo.id(), newInfo.ip(), newInfo.publicIp(), newInfo.cores(), newInfo.diskSize(), newInfo.memorySize(), newInfo.maxOpenFiles(), newInfo.jvm(), newInfo.os());
	}

	private ComputerAssertion computerInfo() {
		Activity.System system = context.system();
		String name = system.name();
		if (name == null || name.isEmpty()) return null;
		Activity.System.Network network = system.network();
		Activity.System.HDD hdd = system.hdd();
		String osName = osName(system);
		return (ComputerAssertion) new ComputerAssertion(context.observer() + "." + name, name).
				ts(Instant.now())
				.isp(network.ISP())
				.os(osName)
				.maxOpenFiles(hdd.maxOpenFiles())
				.cores(system.cpu().cores())
				.diskSize(hdd.capacityMB())
				.memorySize(system.ram().capacityMB())
				.architecture(System.getProperty("os.arch"))
				.jvm(findInstalledJVM())
				.publicIp(network.publicIP())
				.ip(List.of(network.localIP()))
				.observer(box.observer());
	}

	private static String osName(Activity.System system) {
		try {
			return system.operatingSystem().fullName();
		} catch (Exception e) {
			return null;
		}
	}

	private String findInstalledJVM() {
		try {
			Process java = new ProcessBuilder("java", "--version").start();
			Thread.sleep(1000);
			BufferedReader reader = java.inputReader();
			List<String> lines = reader.lines().toList();
			reader.close();
			if (!lines.isEmpty()) return lines.get(0);
		} catch (IOException | InterruptedException e) {
			Logger.error(e);
		}
		return null;
	}
}