package io.intino.consul.container.box;

import io.intino.alexandria.jms.ConnectionConfig;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.actions.ObserverAssertionAction;
import io.intino.consul.container.box.activity.ActivityManager;
import io.intino.consul.container.box.os.local.LocalSystem;
import io.intino.consul.container.box.os.remote.RemoteSystem;
import io.intino.consul.container.box.service.ServiceConfigurator;
import io.intino.consul.framework.Activity;
import io.intino.consul.framework.Activity.Context;
import io.intino.consul.framework.utils.Utils;
import io.intino.consul.terminal.Terminal;

import java.io.File;
import java.util.concurrent.TimeUnit;

import static io.intino.alexandria.logger4j.Logger.setLevel;
import static org.apache.log4j.Level.ERROR;

public class ContainerBox extends AbstractBox {
	private final ContainerStore store;
	private final Terminal terminal;
	private final ActivityManager manager;
	private final Context context;
	private final String hostName;

	public ContainerBox(ContainerConfiguration configuration) throws Exception {
		super(configuration);
		setLevel(ERROR);
		Activity.System system = system();
		hostName = system.name();
		terminal = initTerminal(hostName);
		store = new ContainerStore(configuration.store() == null ? new File(configuration.home(), ".consul") : new File(configuration.store()));
		context = new Context(terminal, system, hostName, observer(), unit(), configuration.args());
		this.manager = new ActivityManager(configuration, context, store, system);
	}

	@Override
	public io.intino.alexandria.core.Box put(Object o) {
		super.put(o);
		return this;
	}

	public ActivityManager activities() {
		return manager;
	}

	public void beforeStart() {
	}

	public void afterStart() {
		terminal.start();
		new ServiceConfigurator(this).configure(terminal.service());
		manager.loadActivitiesFromStore();
		new ObserverAssertionAction(this, context).execute();
		Logger.info("Started Consul v. " + Utils.currentVersion() + " in server: " + hostName);
	}

	public Context context() {
		return context;
	}

	public String observer() {
		return unit() + "consul-" + hostName;
	}

	private String unit() {
		return configuration.unit() == null ? "" : configuration.unit();
	}

	private Activity.System system() throws Exception {
		return configuration.remoteServer() != null ?
				new RemoteSystem(configuration.privateKey(), configuration.remoteServer(), configuration.remotePort(), configuration().remoteUser()) :
				new LocalSystem(configuration.ispProviderToken());
	}

	public void beforeStop() {
		manager.stopActivities();
	}

	public void afterStop() {
		terminal.stop();
		save();
		Logger.info("Shutting down Consul");
	}

	private Terminal initTerminal(String hostName) {
		return new Terminal(observer(), new ConnectionConfig(configuration.datahubUrl(), configuration.datahubUser(), configuration.datahubPassword(), hostName,
				configuration.keyStoreFile(), configuration.trustStoreFile(),
				configuration.keyStorePassword(), configuration.trustStorePassword(), 1, TimeUnit.MINUTES));
	}

	public Activity.Store store() {
		return store;
	}

	public void save() {
		manager.saveConfigurations();
		store.save();
	}
}