package io.intino.consul.container.box.activity;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.schemas.Manifest;
import io.intino.consul.container.box.schemas.Manifest.Service.Endpoint;
import io.intino.consul.container.box.schemas.Manifest.Service.Endpoint.Response;
import io.intino.consul.container.box.schemas.Parameter;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static io.intino.consul.container.box.schemas.Manifest.Service;
import static io.intino.consul.container.box.schemas.Parameter.Type;

public class ManifestReader {
	public static Manifest of(String serialized) {
		try {
			DocumentBuilder documentBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			Document document = documentBuilder.parse(new ByteArrayInputStream(serialized.getBytes()));
			document.getDocumentElement().normalize();
			Element activity = (Element) document.getFirstChild();
			return new Manifest().name(childOf(activity, "name"))
					.description(childOf(activity, "description"))
					.container(containerFrom(asList(activity.getElementsByTagName("container"))))
					.needInstallToConfig(Boolean.getBoolean(childOf(activity, "installToConfig")))
					.dependantActivities(content(asList(activity.getElementsByTagName("dependantActivity"))))
					.parameters(parameters(asList(activity.getElementsByTagName("parameters"))))
					.service(service(asList(activity.getElementsByTagName("service"))));
		} catch (ParserConfigurationException | SAXException | IOException e) {
			Logger.error(e);
		}
		return null;
	}

	private static Manifest.Container containerFrom(List<Node> containerList) {
		if (containerList.isEmpty()) return null;
		Element container = (Element) containerList.get(0);
		String minVersion = childOf(container, "minVersion");
		String maxVersion = childOf(container, "maxVersion");
		return new Manifest.Container().minVersion(minVersion).maxVersion(maxVersion);
	}

	private static List<String> content(List<Node> dependantActivities) {
		return dependantActivities.stream().map(Node::getTextContent).toList();
	}

	private static String childOf(Element activity, String child) {
		return asList(activity.getElementsByTagName(child)).stream()
				.map(node -> node.getTextContent().trim())
				.findFirst()
				.orElse(null);
	}

	private static Service service(List<Node> nodes) {
		if (nodes.isEmpty()) return null;
		Element element = (Element) nodes.get(0);
		return new Service().endpointList(asList(element.getElementsByTagName("endpoint")).stream()
				.map((Node e) -> endPoint((Element) e))
				.toList());
	}

	private static Endpoint endPoint(Element e) {
		return new Endpoint()
				.name(childOf(e, "name"))
				.label(childOf(e, "label"))
				.icon(childOf(e, "icon"))
				.description(childOf(e, "description"))
				.parameters(parameters(asList(e.getElementsByTagName("parameters"))))
				.response(response(asList(e.getElementsByTagName("response"))));
	}

	private static Response response(List<Node> children) {
		Node response = findNodeByName(children, "response").findFirst().orElse(null);
		if (response == null) return null;
		Map<String, String> attributes = asMap(response.getAttributes());
		return new Response().type(attributes.get("type")).description(attributes.get("description"));
	}

	private static List<Parameter> parameters(List<Node> parameters) {
		if (parameters.isEmpty()) return Collections.emptyList();
		return asList(((Element) parameters.get(0)).getElementsByTagName("parameter")).stream().map(p -> {
			Map<String, String> attrs = asMap(p.getAttributes());
			List<String> tsvColumns = attrs.containsKey("columns") ?
					Arrays.stream(attrs.get("columns").split(",")).toList() :
					List.of();
			return new Parameter()
					.name(p.getTextContent().trim())
					.description(attrs.get("description"))
					.isRequired("true".equals(attrs.get("required")))
					.type(Type.valueOf(attrs.get("type").toLowerCase()))
					.tsvColumns(tsvColumns)
					.tsvEditableColumns(attrs.containsKey("editable") ?
							Arrays.stream(attrs.get("editable").split(",")).toList() :
							tsvColumns)
					;

		}).toList();
	}

	private static Map<String, String> asMap(NamedNodeMap attributes) {
		return IntStream.range(0, attributes.getLength())
				.mapToObj(i -> (Attr) attributes.item(i))
				.collect(Collectors.toMap(Attr::getName, Attr::getValue));

	}

	private static Stream<Node> findNodeByName(List<Node> nodes, String name) {
		return nodes.stream().filter(n -> n.getNodeName().equalsIgnoreCase(name));
	}

	private static List<Node> asList(NodeList childNodes) {
		return IntStream.range(0, childNodes.getLength()).mapToObj(childNodes::item).toList();
	}

}
