package io.intino.consul.container.box.service.requests;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.ContainerBox;
import io.intino.consul.container.box.ContainerStore;
import io.intino.consul.container.box.actions.ObserverAssertionAction;
import io.intino.consul.framework.Activity;
import jakarta.jms.BytesMessage;
import jakarta.jms.JMSException;
import jakarta.jms.Message;

public class UpgradeActivityRequest extends ContainerRequest {
	public static final String ID = "upgradeActivity";
	private final ContainerBox box;
	private final Activity.Context context;
	private final ContainerStore store;

	public UpgradeActivityRequest(ContainerBox box, Activity.Context context, ContainerStore store) {
		this.box = box;
		this.context = context;
		this.store = store;
	}

	@Override
	public String id() {
		return ID;
	}

	@Override
	public RequestResult responseTo(Message request) {
		BytesMessage message = (BytesMessage) request;
		try {
			String newActivity = activity(request);
			Logger.info("Upgrading activity to " + newActivity);
			if (box.activities().enabledActivities().contains(newActivity)) {
				Logger.info("Activity already updated: " + newActivity);
				return new RequestResult(true, "Activity already updated: " + newActivity);
			}
			String lastActivity = lastActivity(newActivity);
			if (lastActivity == null) {
				Logger.info("Activity not found: " + newActivity);
				return new RequestResult(false, "Activity not found: " + newActivity);
			}
			if (lastActivity.equals(newActivity)) return new RequestResult(true, "Activity already installed");
			install(message, newActivity);
			copyConfiguration(lastActivity, newActivity);
			Activity.Result result;
			if (box.activities().enabledActivities().contains(lastActivity)) {
				result = box.activities().disableActivity(lastActivity);
				if (!result.success()) return new RequestResult(false, result.remarks());
			}
			result = box.activities().uninstallActivity(lastActivity);
			if (!result.success()) return new RequestResult(false, result.remarks());
			result = box.activities().enableActivity(newActivity);
			new Thread(() -> new ObserverAssertionAction(box, context).execute(), newActivity + "-upgrade").start();
			Logger.info("Upgraded activity to " + newActivity);
			return new RequestResult(result.success(), result.remarks());
		} catch (Throwable e) {
			Logger.error(e);
			return new RequestResult(false, e.getMessage());
		}
	}

	@Override
	public boolean isAvailable(Message message) {
		return true;
	}

	private void copyConfiguration(String last, String newActivity) {
		if (last == null) return;
		store.save();
		box.activities().updateConfiguration(newActivity, store.activityParams(last));
	}

	private String lastActivity(String artifact) {
		String[] split = artifact.split(":");
		String id = split[0] + ":" + split[1];
		return box.activities().activities().keySet().stream().filter(a -> a.startsWith(id + ":")).findFirst().orElse(null);
	}

	private void install(BytesMessage message, String artifact) throws JMSException {
		byte[] payload = new byte[(int) message.getBodyLength()];
		message.readBytes(payload);
		box.activities().installActivity(artifact, payload);
	}
}