package io.intino.consul.container.box.os.remote.linux;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;
import oshi.annotation.concurrent.ThreadSafe;
import oshi.driver.linux.proc.ProcessStat;
import oshi.util.ParseUtil;
import oshi.util.UserGroupInfo;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static io.intino.consul.container.box.os.remote.linux.Util.isBlank;
import static io.intino.consul.container.box.os.remote.linux.Util.parseLongOrDefault;
import static java.lang.String.format;
import static oshi.util.Memoizer.memoize;
import static oshi.util.ParseUtil.whitespaces;

@ThreadSafe
public class LinuxOSProcess implements Activity.System.OSProcess {
	private static final int[] PROC_PID_STAT_ORDERS = new int[ProcPidStat.values().length];

	private final File proc;
	private final Activity.System.FileSystem fileSystem;
	private final Supplier<Integer> bitness = memoize(this::queryBitness);
	private final Supplier<String> commandLine = memoize(this::queryCommandLine);
	private final Supplier<List<String>> arguments = memoize(this::queryArguments);
	private String name;
	private String path = "";
	private String user;
	private String group;
	private State state = State.INVALID;
	private final boolean isOSProcess;
	private int parentProcessID;
	private int threadCount;
	private long virtualSize;
	private long residentSetSize;
	private long kernelTime;
	private long userTime;
	private long startTime;
	private long upTime;
	private long bytesRead;
	private long bytesWritten;

	public LinuxOSProcess(File proc, Activity.System.OperatingSystem os, Set<String> systemProcesses) throws Exception {
		this.proc = proc;
		this.fileSystem = os.fileSystem();
		loadAttributes(os, os.processRunner());
		this.isOSProcess = systemProcesses.contains(name) || systemProcesses.contains(name.split("/")[0]);
	}

	@Override
	public String name() {
		return this.name;
	}

	@Override
	public String path() {
		return this.path;
	}

	@Override
	public String commandLine() {
		return commandLine.get();
	}

	private String queryCommandLine() {
		String pidCmd = fileSystem.readFile(format(ProcPath.PID_CMDLINE, processID()));
		if (pidCmd == null) return "";
		return String.join(" ", pidCmd.split("\0"));
	}

	@Override
	public List<String> arguments() {
		return arguments.get();
	}

	private List<String> queryArguments() {
		return Collections.unmodifiableList(ParseUtil
				.parseByteArrayToStrings(fileSystem.readFileBytes(format(ProcPath.PID_CMDLINE, processID()))));
	}

	@Override
	public String user() {
		return this.user;
	}

	@Override
	public String group() {
		return this.group;
	}

	@Override
	public State state() {
		return this.state;
	}

	@Override
	public boolean isOSProcess() {
		return isOSProcess;
	}

	@Override
	public boolean isSystemService() {
		return false;//TODO
	}

	@Override
	public String systemServiceName() {
		return null;//TODO
	}

	@Override
	public int processID() {
		return Integer.parseInt(proc.getName());
	}

	@Override
	public int parentProcessID() {
		return this.parentProcessID;
	}

	@Override
	public int threadCount() {
		return this.threadCount;
	}

	@Override
	public long virtualSize() {
		return this.virtualSize;
	}

	@Override
	public long residentSetSize() {
		return this.residentSetSize;
	}

	@Override
	public long kernelTime() {
		return this.kernelTime;
	}

	@Override
	public long userTime() {
		return this.userTime;
	}

	@Override
	public long upTime() {
		return this.upTime;
	}

	@Override
	public long startTime() {
		return this.startTime;
	}

	@Override
	public long bytesRead() {
		return this.bytesRead;
	}

	@Override
	public long bytesWritten() {
		return this.bytesWritten;
	}

	@Override
	public long openFiles() {
		return ProcessStat.getFileDescriptorFiles(processID()).length;
	}

	public long openFilesLimit() {
		return getProcessOpenFileLimit(processID());
	}

	@Override
	public double processCpuLoadCumulative() {
		return 0;
	}

	@Override
	public int getBitness() {
		return this.bitness.get();
	}

	private int queryBitness() {
		byte[] buffer = new byte[5];
		if (!path.isEmpty()) {
			try (InputStream is = new FileInputStream(path)) {
				if (is.read(buffer) == buffer.length) {
					return buffer[4] == 1 ? 32 : 64;
				}
			} catch (IOException e) {
				Logger.warn("Failed to read process file: " + path);
			}
		}
		return 0;
	}

	private void loadAttributes(Activity.System.OperatingSystem os, Activity.System.ProcessRunner processRunner) throws Exception {
		this.path = exeFile(processRunner);
		Map<String, String> io = readAsMap(format(ProcPath.PID_IO, processID()));
		Map<String, String> status = readAsMap(format(ProcPath.PID_STATUS, processID()));
		String stat = fileSystem.readFile(format(ProcPath.PID_STAT, processID()));
		this.name = status.getOrDefault("Name", "").trim();
		this.state = Util.getState(status.getOrDefault("State", "U").charAt(0));
		String userID = whitespaces.split(status.getOrDefault("Uid", ""))[0];
		this.user = UserGroupInfo.getUser(userID).trim();
		String groupID = whitespaces.split(status.getOrDefault("Gid", ""))[0];
		this.group = UserGroupInfo.getGroupName(groupID);
		if (stat == null || stat.isEmpty()) {
			this.state = State.INVALID;
			return;
		}
		getMissingDetails(status, stat);
		long now = System.currentTimeMillis();
		long[] statArray = ParseUtil.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, ProcessStat.PROC_PID_STAT_LENGTH, ' ');
		this.startTime = startTime(os);
		this.parentProcessID = (int) statArray[ProcPidStat.PPID.ordinal()];
		this.threadCount = (int) statArray[ProcPidStat.THREAD_COUNT.ordinal()];
		this.virtualSize = statArray[ProcPidStat.VSZ.ordinal()];
		this.residentSetSize = statArray[ProcPidStat.RSS.ordinal()] * os.pageSize();
		this.kernelTime = statArray[ProcPidStat.KERNEL_TIME.ordinal()] * 1000L / os.userHz();
		this.userTime = statArray[ProcPidStat.USER_TIME.ordinal()] * 1000L / os.userHz();
		this.upTime = now - this.startTime;
		this.bytesRead = parseLongOrDefault(io.getOrDefault("read_bytes", ""), 0L);
		this.bytesWritten = parseLongOrDefault(io.getOrDefault("write_bytes", ""), 0L);
	}

	private long startTime(Activity.System.OperatingSystem os) throws Exception {
		String time = os.processRunner().execute("awk -v ticks=\"$(getconf CLK_TCK)\" -v epoch=\"$(date +%s)\" '\n" +
				"  NR==1 { now=$1; next }\n" +
				"  END { printf \"%9.0f\\n\", epoch - (now-($20/ticks)) }' /proc/uptime RS=')' /proc/" + proc.getName() + "/stat |\n" +
				"xargs -i date -d @{}");
		TemporalAccessor parse = DateTimeFormatter.ofPattern("EEE LLL d hh:mm:ss a yyyy", Locale.US).parse(time.replace(" UTC ", " ").replace("  ", " ").trim());
		return ZonedDateTime.of(LocalDateTime.from(parse), ZoneId.of("UTC")).toInstant().toEpochMilli();
	}

	private Map<String, String> readAsMap(String path) {
		String content = fileSystem.readFile(path);
		if (content == null) return new HashMap<>();
		return content.lines().map(l -> l.split(":")).collect(Collectors.toMap(s -> s[0], s -> s[1]));
	}

	private String exeFile(Activity.System.ProcessRunner runner) {
		String path = "";
		String procPidExe = format(ProcPath.PID_EXE, processID());
		try {
			Path link = Paths.get(procPidExe);
			path = runner.execute("readlink", "-f", link.toString());
			int index = path.indexOf(" (deleted)");
			if (index != -1) path = path.substring(0, index);
		} catch (Exception e) {
			Logger.debug("Unable to open symbolic link " + procPidExe + ". " + e.getMessage());
		}
		return path;
	}

	private static void getMissingDetails(Map<String, String> status, String stat) {
		if (status == null || stat == null) return;
		int nameStart = stat.indexOf('(');
		int nameEnd = stat.indexOf(')');
		if (isBlank(status.get("Name")) && nameStart > 0 && nameStart < nameEnd)
			status.put("Name", stat.substring(nameStart + 1, nameEnd));
		if (isBlank(status.get("State")) && nameEnd > 0 && stat.length() > nameEnd + 2)
			status.put("State", String.valueOf(stat.charAt(nameEnd + 2)));
	}

	private long getProcessOpenFileLimit(long processId) {
		String limitsPath = format(ProcPath.PID_LIMITS, processId);
		if (!fileSystem.exists(limitsPath)) return -1;
		return fileSystem.readFile(limitsPath).lines()
				.filter(line -> line.startsWith("Max open files"))
				.findFirst()
				.map(s -> Long.parseLong(s.split("\\D+")[2]))
				.orElse(-1L);
	}

	static {
		for (ProcPidStat stat : ProcPidStat.values()) {
			PROC_PID_STAT_ORDERS[stat.ordinal()] = stat.getOrder() - 1;
		}
	}


	private enum ProcPidStat {
		PPID(4), MINOR_FAULTS(10), MAJOR_FAULTS(12), USER_TIME(14), KERNEL_TIME(15), PRIORITY(18), THREAD_COUNT(20),
		START_TIME(22), VSZ(23), RSS(24);

		private final int order;

		public int getOrder() {
			return this.order;
		}

		ProcPidStat(int order) {
			this.order = order;
		}
	}
}
