package io.intino.consul.container.box.os.remote;

import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;

import java.io.File;
import java.util.Arrays;

public class RemoteSystem implements Activity.System {
	private final Session session;
	private final RemoteOperatingSystem operatingSystem;

	public RemoteSystem(String privateKey, String remoteServer, String remotePort, String remoteUser) throws Exception {
		this.session = createSession(privateKey, remoteServer, remotePort, remoteUser);
		operatingSystem = new RemoteOperatingSystem(session);
		Logger.info("Connected to remote system!");
	}

	@Override
	public String name() {
		try {
			return new RemoteProcessRunner(session).execute("hostname").trim();
		} catch (Exception e) {
			Logger.error(e.getMessage());
			return "consul-default-server";
		}
	}

	@Override
	public OperatingSystem operatingSystem() {
		return operatingSystem;
	}

	@Override
	public HDD hdd() {
		try {
			String executed = new RemoteProcessRunner(session).execute("df -m .").trim();
			String[] values = Arrays.stream(executed.split("\n")[1].split(" ")).map(String::trim).filter(f -> !f.isEmpty()).toArray(String[]::new);
			String openFiles = new RemoteProcessRunner(session).execute("ulimit -n").trim();
			return new HDD(Integer.parseInt(values[1]), Integer.parseInt(values[2]), Integer.parseInt(openFiles.trim()));
		} catch (Exception e) {
			Logger.error(e.getMessage());
			return new HDD(0, 0, 0);
		}
	}

	@Override
	public RAM ram() {
		try {
			String ram = new RemoteProcessRunner(session).execute("free -m").trim();
			String[] values = Arrays.stream(ram.split("\n")[1].split(" ")).map(String::trim).filter(f -> !f.isEmpty()).toArray(String[]::new);
			return new RAM(Integer.parseInt(values[1]), Integer.parseInt(values[2]));
		} catch (Exception e) {
			Logger.error(e.getMessage());
			return new RAM(0, 0);
		}
	}

	@Override
	public CPU cpu() {
		try {
			String cores = new RemoteProcessRunner(session).execute("cat /proc/cpuinfo | grep processor | wc -l").trim();
			String[] iostat = new RemoteProcessRunner(session).execute("iostat -c").trim().split("\n");
			String[] values = Arrays.stream(iostat[iostat.length - 1].split(" ")).map(String::trim).filter(f -> !f.isEmpty()).toArray(String[]::new);
			return new CPU(Integer.parseInt(cores), (long) Double.parseDouble(values[0]), 0);
		} catch (Exception e) {
			Logger.error("Command: 'iostat -c', message: " + e.getMessage());
			return new CPU(0, 0, 0);
		}
	}

	@Override
	public Network network() {
		try {
			String ip = new RemoteProcessRunner(session).execute("hostname -I").trim();
			if (ip.contains(" ")) ip = ip.substring(0, ip.indexOf(" "));
			return new Network(ip, null, null);
		} catch (Exception e) {
			Logger.error(e.getMessage());
			return new Network(null, null, null);
		}
	}

	protected Session createSession(String privateKey, String remoteServer, String remotePort, String remoteUser) throws JSchException {
		JSch jSch = new JSch();
		jSch.addIdentity(privateKey);
		jSch.setKnownHosts(new File(new File(privateKey).getAbsoluteFile(), "known_hosts").getAbsolutePath());
		Session session = jSch.getSession(remoteUser, remoteServer, Integer.parseInt(remotePort));
		session.setConfig("StrictHostKeyChecking", "no");
		session.setServerAliveInterval(30000);
		session.setConfig("PreferredAuthentications", "publickey,password");
		session.setConfig("useDNS", "no");
		session.setTimeout(60_000);
		session.connect(60_000);
		return session;
	}
}
