package io.intino.consul.container.box.os.remote;

import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.os.remote.linux.LinuxOSProcess;
import io.intino.consul.container.box.os.remote.linux.ProcPath;
import io.intino.consul.framework.Activity;
import io.intino.consul.framework.Activity.System.FileSystem;
import io.intino.consul.framework.Activity.System.Measurements;
import io.intino.consul.framework.Activity.System.OSProcess;
import io.intino.consul.framework.Activity.System.ProcessRunner;
import oshi.driver.linux.proc.Auxv;
import oshi.driver.linux.proc.UpTime;
import oshi.util.FileUtil;
import oshi.util.ParseUtil;

import java.io.*;
import java.nio.ByteBuffer;
import java.util.*;
import java.util.regex.Pattern;

import static io.intino.consul.framework.utils.Utils.isNumber;
import static java.util.stream.Collectors.toSet;
import static oshi.util.ParseUtil.parseLongOrDefault;

public class RemoteOperatingSystem implements Activity.System.OperatingSystem {
	private final RemoteProcessRunner processRunner;
	private final Session session;
	private final RemoteFileSystem remoteFileSystem;
	private final long bootTime;
	private long userHz;
	private long pageSize;
	private static Set<String> systemProcesses;
	private String name;

	public RemoteOperatingSystem(Session session) {
		this.session = session;
		this.remoteFileSystem = new RemoteFileSystem(session);
		this.processRunner = new RemoteProcessRunner(session);
		this.bootTime = booTime();
		if (systemProcesses == null || systemProcesses.isEmpty()) systemProcesses = systemProcesses();
		try {
			name = processRunner().execute("uname -o").trim();
		} catch (Exception ignored) {
		}
		updateProperties();
	}

	@Override
	public String name() {
		return name;
	}

	@Override
	public String fullName() {
		return name;
	}

	@Override
	public long bootTimeSeconds() {
		return this.bootTime;
	}

	@Override
	public FileSystem fileSystem() throws IOException {
		return remoteFileSystem;
	}

	@Override
	public ProcessRunner processRunner() {
		return processRunner;
	}

	@Override
	public int createConnection(int port) {
		synchronized (session) {
			try {
				if (contains(port)) return port;
				return session.setPortForwardingL(port, "localhost", port);
			} catch (JSchException e) {
				Logger.error(e.getMessage());
				return 0;
			}
		}
	}

	private boolean contains(int port) {
		try {
			return Arrays.stream(session.getPortForwardingL()).anyMatch(l -> l.contains(port + ":"));
		} catch (JSchException e) {
			Logger.error(e);
			return false;
		}
	}

	@Override
	public void removeConnection(int port) {
		synchronized (session) {
			try {
				if (!contains(port)) session.delPortForwardingL(port);
			} catch (JSchException e) {
				Logger.error(e.getMessage());
			}
		}
	}

	@Override
	public List<? extends OSProcess> processes() throws IOException {
		return getPidFiles().stream()
				.map(this::linuxOSProcess)
				.filter(Objects::nonNull)
				.toList();
	}

	private LinuxOSProcess linuxOSProcess(File f) {
		try {
			return new LinuxOSProcess(f, this, systemProcesses);
		} catch (Exception e) {
			Logger.error(f.getPath() + ": " + e.getMessage());
			return null;
		}
	}

	private List<File> getPidFiles() throws IOException {
		File directory = new File(ProcPath.PROC);
		return fileSystem().listDirectory(directory.getAbsolutePath()).stream()
				.filter(file -> DIGITS.matcher(file).matches())
				.map(n -> new File(directory, n))
				.toList();
	}

	public static final Pattern DIGITS = Pattern.compile("\\d+");

	@Override
	public int processCount() {
		return 0;
	}

	private long booTime() {
		long tempBT = bootTimeFromProc();
		if (tempBT == 0) tempBT = System.currentTimeMillis() / 1000L - (long) UpTime.getSystemUptimeSeconds();
		return tempBT;
	}

	public long bootTimeFromProc() {
		try {
			return fileSystem().readFile(ProcPath.STAT).lines()
					.filter(stat -> stat.startsWith("btime"))
					.findFirst()
					.map(t -> parseLongOrDefault(ParseUtil.whitespaces.split(t)[1], 0L))
					.orElse(0L);
		} catch (IOException e) {
			Logger.error(e.getMessage());
			return 0;
		}
	}

	public long userHz() {
		return this.userHz;
	}

	public long pageSize() {
		return this.pageSize;
	}

	@Override
	public Measurements measurements() {
		String command = String.join(";echo -----;",
				"free -m",
				"df -m .",
				"iostat -c",
				"sysctl fs.file-nr",
				"ps -eo nlwp | tail -n +2 | awk '{ num_threads += $1 } END { print num_threads }'",
				"cat /sys/class/net/eth0/statistics/tx_bytes",
				"cat /sys/class/net/eth0/statistics/rx_bytes",
				"uptime");
		try {
			String[] results = Arrays.stream(processRunner.execute(command).trim().split("-----")).map(String::trim).toArray(String[]::new);
			return new Measurements() {
				@Override
				public long usageRAM() {
					try {
						return usage(results[0]);
					} catch (Exception e) {
						Logger.error(e.getMessage());
						return 0;
					}
				}

				@Override
				public long usageHDD() {
					try {
						return usage(results[1]);
					} catch (Exception e) {
						Logger.error(e.getMessage());
						return 0;
					}
				}

				@Override
				public double usageCPU() {
					try {
						String[] iostat = results[2].trim().split("\n");
						String[] values = Arrays.stream(iostat[iostat.length - 1].split(" "))
								.map(String::trim)
								.filter(f -> !f.isEmpty())
								.toArray(String[]::new);
						if (values.length > 0 && isNumber(values[0])) return (long) Double.parseDouble(values[0]);
					} catch (Exception e) {
						Logger.error(e.getMessage());
					}
					return 0;
				}

				@Override
				public double usageSystem() {
					try {
						String trim = results[7].trim();
						String averages = trim.substring(trim.lastIndexOf(":") + 1);
						Double[] array = Arrays.stream(averages.split(",")).map(String::trim).map(Double::parseDouble).toArray(Double[]::new);
						return array[2];
					} catch (Throwable e) {
						Logger.error(e);
						return 0;
					}
				}

				@Override
				public int usageFiles() {
					try {
						String[] files = Arrays.stream(results[3].split(" "))
								.map(String::trim)
								.filter(f -> !f.isEmpty() || f.isBlank()).toArray(String[]::new);
						if (files.length == 0) return 0;
						String[] used = files[2].split("\t");
						if (used.length > 0 && isNumber(used[0])) return Integer.parseInt(used[0]);
					} catch (Exception e) {
						Logger.error(e.getMessage());
					}
					return 0;
				}

				@Override
				public long usageThreads() {
					try {
						String result = results[4].trim();
						return (isNumber(result)) ? Integer.parseInt(result) : 0;
					} catch (Exception e) {
						Logger.error(e.getMessage());
						return 0;
					}
				}

				@Override
				public long dataReceived() {
					try {
						String result = results[5].trim();
						return (isNumber(result)) ? Long.parseLong(result) / (1024 * 1024) : 0;
					} catch (Exception e) {
						Logger.error(e.getMessage());
						return 0;
					}
				}

				@Override
				public long dataSent() {
					try {
						String result = results[6].trim();
						return (isNumber(result)) ? Long.parseLong(result) / (1024 * 1024) : 0;
					} catch (Exception e) {
						Logger.error(e.getMessage());
						return 0;
					}
				}

				@Override
				public double temperatureKernel() {
					return 0;
				}

				@Override
				public double temperatureExternal() {
					return 0;
				}
			};
		} catch (Exception e) {
			Logger.error(e);
			return null;
		}
	}

	private static Integer usage(String executed) {
		String[] lines = executed.split("\n");
		if (lines.length < 2) return 0;
		String[] values = Arrays.stream(lines[1].split(" "))
				.map(String::trim)
				.filter(f -> !f.isEmpty())
				.toArray(String[]::new);
		if (values.length >= 3 && isNumber(values[2])) return Integer.parseInt(values[2]);
		return 0;
	}

	private void updateProperties() {
		try {
			Map<Integer, Long> auxv = queryAuxv();
			long hz = auxv.getOrDefault(Auxv.AT_CLKTCK, 0L);
			userHz = hz > 0 ? hz : parseLongOrDefault(processRunner.execute("getconf", "CLK_TCK"), 100L);
			long pageSize = auxv.getOrDefault(Auxv.AT_PAGESZ, 0L);
			this.pageSize = pageSize > 0 ? pageSize : parseLongOrDefault(processRunner.execute("getconf", "PAGE_SIZE"), 4096L);
		} catch (Exception e) {
			Logger.error(e.getMessage());
		}
	}

	private Map<Integer, Long> queryAuxv() {
		try {
			byte[] array = fileSystem().readFileBytes(ProcPath.AUXV);
			ByteBuffer buff = ByteBuffer.wrap(array);
			Map<Integer, Long> auxvMap = new HashMap<>();
			int key;
			do {
				key = FileUtil.readNativeLongFromBuffer(buff).intValue();
				if (key > 0) auxvMap.put(key, FileUtil.readNativeLongFromBuffer(buff).longValue());
			} while (key > 0);
			return auxvMap;
		} catch (IOException e) {
			Logger.error(e);
			return Map.of();
		}
	}

	private Set<String> systemProcesses() {
		InputStream stream = this.getClass().getClassLoader().getResourceAsStream(Name.Unix.name().toLowerCase() + ".system.processes.txt");
		if (stream == null) {
			Logger.error("Resource not found: " + Name.Unix.name().toLowerCase() + ".system.processes.txt");
			return Set.of();
		}
		return new BufferedReader(new InputStreamReader(stream)).lines().collect(toSet());
	}
}