package io.intino.consul.container.box.activity;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.ContainerConfiguration;
import io.intino.consul.container.box.ContainerStore;
import io.intino.consul.container.box.activity.ActivityLoader.JarClassLoader;
import io.intino.consul.framework.Activity;
import io.intino.consul.framework.Activity.Context;
import io.intino.consul.framework.Activity.Result;
import io.intino.consul.framework.utils.Utils;

import java.io.File;
import java.io.IOException;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static io.intino.consul.container.box.ContainerStore.asBooleanMap;

public class ActivityManager {
	public static final String ROOT_ACTIVITIES = "root#activities";
	private final HashMap<String, Activity> enabledActivities;
	private final ContainerConfiguration configuration;
	private final Context context;
	private final ContainerStore store;
	private final Activity.System system;
	private final ActivityLoader loader;

	public ActivityManager(ContainerConfiguration configuration, Context context, ContainerStore store, Activity.System system) {
		this.configuration = configuration;
		this.context = context;
		this.store = store;
		this.system = system;
		this.loader = new ActivityLoader();
		this.enabledActivities = new HashMap<>();
	}

	public List<String> enabledActivities() {
		return new ArrayList<>(enabledActivities.keySet());
	}

	public void loadActivitiesFromStore() {
		Map<String, Boolean> activities = activities();
		if (activities == null) return;
		activities.keySet().stream()
				.filter(activities::get)
				.forEach(this::enableActivity);
	}

	public void stopActivities() {
		enabledActivities.values().forEach(ActivityManager::stop);
	}

	public Result enableActivity(String activityId) {
		Activity activity = enabledActivities.get(activityId);
		if (activity != null) return new Result(true, "Activity is already enable");
		if (!loader.isSuitable(store.activity(activityId), Utils.currentVersion()))
			return new Result(true, "Activity is not suitable with this container. Please upgrade activity to last version");
		activity = loader.load(store.activity(activityId));
		if (activity == null) return new Result(false, "Activity not found");
		Result enabled = configure(activity, store.activityParams(activityId));
		if (!enabled.success()) return enabled;
		enabledActivities.put(activityId, activity);
		update(activityId, true);
		Logger.info("Enabled activity " + activityId);
		new Thread(activity::onStart).start();
		return new Result(true, "Activity " + activityId + " enabled");
	}

	public Result disableActivity(String activityId) {
		Activity activity = enabledActivities.get(activityId);
		if (activity == null) return new Result(false, "Activity not found");
		stop(activity);
		enabledActivities.remove(activityId);
		update(activityId, false);
		Logger.info("Disabled activity " + activityId);
		return new Result(true, "Activity " + activityId + " disabled");
	}

	public Map<String, Boolean> activities() {
		return store.getOrElse(ROOT_ACTIVITIES, asBooleanMap, new HashMap<>());
	}

	public Result updateConfiguration(String activityId, Map<String, String> configuration) {
		if (!activities().containsKey(activityId)) return new Result(false, "Activity not found");
		store.saveActivityParams(activityId, configuration);
		Logger.info("Updated configuration of " + activityId);
		return new Result(true, "Configuration of activity " + activityId + " updated");
	}

	public synchronized Result installActivity(String activityId, byte[] payload) {
		Map<String, Boolean> activities = activities();
		if (!activities.containsKey(activityId)) activities.put(activityId, false);
		store.put(ROOT_ACTIVITIES, activities);
		store.saveActivity(activityId, payload);
		Logger.info("Installed activity " + activityId);
		return new Result(true, "Activity " + activityId + " installed");
	}

	public synchronized Result uninstallActivity(String activityId) {
		Map<String, Boolean> activities = activities();
		if (activities == null) return new Result(false, "Activity not found");
		activities.remove(activityId);
		store.put(ROOT_ACTIVITIES, activities);
		remove(activityId, this.enabledActivities.remove(activityId));
		store.save();
		Logger.info("Uninstalled activity " + activityId);
		return new Result(true, "Activity " + activityId + " uninstalled");
	}

	public synchronized void saveConfigurations() {
		enabledActivities.forEach((k, v) -> store.saveActivityParams(k, v.currentConfiguration()));
	}

	private void update(String activityId, boolean value) {
		Map<String, Boolean> activities = activities();
		activities.put(activityId, value);
		store.put(ROOT_ACTIVITIES, activities);
	}

	private void remove(String activityId, Activity activity) {
		if (activity != null) {
			stop(activity);
			stopActivityThreads(activityId);
			closeClassLoader(activity);
			java.lang.System.gc();
		}
		clearStore(activityId);
		store.removeActivity(activityId);
	}

	private void stopActivityThreads(String artifact) {
		Thread.getAllStackTraces().keySet().stream()
				.filter(t -> t.getContextClassLoader() instanceof URLClassLoader &&
						new File(((URLClassLoader) t.getContextClassLoader()).getURLs()[0].getFile()).equals(store.activity(artifact)))
				.forEach(Thread::interrupt);
	}

	private Result configure(Activity a, Map<String, String> args) {
		try {
			return a.configure(new Context(context.terminal(), system, context.hostName(), context.observer(), context.ss(), mergeArgs(args, context.initialConfiguration())), new ActivityStore(a.id(), store));
		} catch (Throwable e) {
			Logger.error(e);
			return new Result(false, e.getMessage());
		}
	}

	private Map<String, String> mergeArgs(Map<String, String> params, Map<String, String> args) {
		HashMap<String, String> merged = new HashMap<>(params);
		merged.putAll(args);
		return merged;
	}

	private static void stop(Activity activity) {
		try {
			activity.onStop();
		} catch (Throwable e) {
			Logger.error(e);
		}
	}

	private void clearStore(String id) {
		store.removeAll(k -> k.startsWith(id + Activity.Store.Separator));
	}

	private static void closeClassLoader(Activity activity) {
		ClassLoader classLoader = activity.getClass().getClassLoader();
		if (classLoader instanceof JarClassLoader) {
			classLoader.clearAssertionStatus();
			try {
				((JarClassLoader) classLoader).close();
			} catch (IOException e) {
				Logger.error(e);
			}
		}
	}
}