/*
 * Decompiled with CFR 0.152.
 */
package io.intino.consul.container.box.os.remote.linux;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.os.remote.linux.Util;
import io.intino.consul.framework.Activity;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import oshi.annotation.concurrent.ThreadSafe;
import oshi.driver.linux.proc.ProcessStat;
import oshi.util.Memoizer;
import oshi.util.ParseUtil;
import oshi.util.UserGroupInfo;

@ThreadSafe
public class LinuxOSProcess
implements Activity.System.OSProcess {
    private static final int[] PROC_PID_STAT_ORDERS = new int[ProcPidStat.values().length];
    private final File proc;
    private final Activity.System.FileSystem fileSystem;
    private final Supplier<Integer> bitness = Memoizer.memoize(this::queryBitness);
    private final Supplier<String> commandLine = Memoizer.memoize(this::queryCommandLine);
    private final Supplier<List<String>> arguments = Memoizer.memoize(this::queryArguments);
    private String name;
    private String path = "";
    private String user;
    private String group;
    private Activity.System.OSProcess.State state = Activity.System.OSProcess.State.INVALID;
    private final boolean isOSProcess;
    private int parentProcessID;
    private int threadCount;
    private long virtualSize;
    private long residentSetSize;
    private long kernelTime;
    private long userTime;
    private long startTime;
    private long upTime;
    private long bytesRead;
    private long bytesWritten;

    public LinuxOSProcess(File proc, Activity.System.OperatingSystem os, Set<String> systemProcesses) throws Exception {
        this.proc = proc;
        this.fileSystem = os.fileSystem();
        this.loadAttributes(os, os.processRunner());
        this.isOSProcess = systemProcesses.contains(this.name) || systemProcesses.contains(this.name.split("/")[0]);
    }

    @Override
    public String name() {
        return this.name;
    }

    @Override
    public String path() {
        return this.path;
    }

    @Override
    public String commandLine() {
        return this.commandLine.get();
    }

    private String queryCommandLine() {
        String pidCmd = this.fileSystem.readFile(String.format("/proc/%d/cmdline", this.processID()));
        if (pidCmd == null) {
            return "";
        }
        return String.join((CharSequence)" ", pidCmd.split("\u0000"));
    }

    @Override
    public List<String> arguments() {
        return this.arguments.get();
    }

    private List<String> queryArguments() {
        return Collections.unmodifiableList(ParseUtil.parseByteArrayToStrings(this.fileSystem.readFileBytes(String.format("/proc/%d/cmdline", this.processID()))));
    }

    @Override
    public String user() {
        return this.user;
    }

    @Override
    public String group() {
        return this.group;
    }

    @Override
    public Activity.System.OSProcess.State state() {
        return this.state;
    }

    @Override
    public boolean isOSProcess() {
        return this.isOSProcess;
    }

    @Override
    public boolean isSystemService() {
        return false;
    }

    @Override
    public String systemServiceName() {
        return null;
    }

    @Override
    public int processID() {
        return Integer.parseInt(this.proc.getName());
    }

    @Override
    public int parentProcessID() {
        return this.parentProcessID;
    }

    @Override
    public int threadCount() {
        return this.threadCount;
    }

    @Override
    public long virtualSize() {
        return this.virtualSize;
    }

    @Override
    public long residentSetSize() {
        return this.residentSetSize;
    }

    @Override
    public long kernelTime() {
        return this.kernelTime;
    }

    @Override
    public long userTime() {
        return this.userTime;
    }

    @Override
    public long upTime() {
        return this.upTime;
    }

    @Override
    public long startTime() {
        return this.startTime;
    }

    @Override
    public long bytesRead() {
        return this.bytesRead;
    }

    @Override
    public long bytesWritten() {
        return this.bytesWritten;
    }

    @Override
    public long openFiles() {
        return ProcessStat.getFileDescriptorFiles(this.processID()).length;
    }

    @Override
    public long openFilesLimit() {
        return this.getProcessOpenFileLimit(this.processID());
    }

    @Override
    public double processCpuLoadCumulative() {
        return 0.0;
    }

    @Override
    public int getBitness() {
        return this.bitness.get();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private int queryBitness() {
        byte[] buffer = new byte[5];
        if (this.path.isEmpty()) return 0;
        try (FileInputStream is = new FileInputStream(this.path);){
            if (((InputStream)is).read(buffer) != buffer.length) return 0;
            int n = buffer[4] == 1 ? 32 : 64;
            return n;
        }
        catch (IOException e) {
            Logger.warn("Failed to read process file: " + this.path);
        }
        return 0;
    }

    private void loadAttributes(Activity.System.OperatingSystem os, Activity.System.ProcessRunner processRunner) throws Exception {
        this.path = this.exeFile(processRunner);
        Map<String, String> io = this.readAsMap(String.format("/proc/%d/io", this.processID()));
        Map<String, String> status = this.readAsMap(String.format("/proc/%d/status", this.processID()));
        String stat = this.fileSystem.readFile(String.format("/proc/%d/stat", this.processID()));
        this.name = status.getOrDefault("Name", "").trim();
        this.state = Util.getState(status.getOrDefault("State", "U").charAt(0));
        String userID = ParseUtil.whitespaces.split(status.getOrDefault("Uid", ""))[0];
        this.user = UserGroupInfo.getUser(userID).trim();
        String groupID = ParseUtil.whitespaces.split(status.getOrDefault("Gid", ""))[0];
        this.group = UserGroupInfo.getGroupName(groupID);
        if (stat == null || stat.isEmpty()) {
            this.state = Activity.System.OSProcess.State.INVALID;
            return;
        }
        LinuxOSProcess.getMissingDetails(status, stat);
        long now = System.currentTimeMillis();
        long[] statArray = ParseUtil.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, ProcessStat.PROC_PID_STAT_LENGTH, ' ');
        this.startTime = this.startTime(os);
        this.parentProcessID = (int)statArray[ProcPidStat.PPID.ordinal()];
        this.threadCount = (int)statArray[ProcPidStat.THREAD_COUNT.ordinal()];
        this.virtualSize = statArray[ProcPidStat.VSZ.ordinal()];
        this.residentSetSize = statArray[ProcPidStat.RSS.ordinal()] * os.pageSize();
        this.kernelTime = statArray[ProcPidStat.KERNEL_TIME.ordinal()] * 1000L / os.userHz();
        this.userTime = statArray[ProcPidStat.USER_TIME.ordinal()] * 1000L / os.userHz();
        this.upTime = now - this.startTime;
        this.bytesRead = Util.parseLongOrDefault(io.getOrDefault("read_bytes", ""), 0L);
        this.bytesWritten = Util.parseLongOrDefault(io.getOrDefault("write_bytes", ""), 0L);
    }

    private long startTime(Activity.System.OperatingSystem os) throws Exception {
        String time = os.processRunner().execute("awk -v ticks=\"$(getconf CLK_TCK)\" -v epoch=\"$(date +%s)\" '\n  NR==1 { now=$1; next }\n  END { printf \"%9.0f\\n\", epoch - (now-($20/ticks)) }' /proc/uptime RS=')' /proc/" + this.proc.getName() + "/stat |\nxargs -i date -d @{}");
        TemporalAccessor parse = DateTimeFormatter.ofPattern("EEE LLL d hh:mm:ss a yyyy", Locale.US).parse(time.replace(" UTC ", " ").replace("  ", " ").trim());
        return ZonedDateTime.of(LocalDateTime.from(parse), ZoneId.of("UTC")).toInstant().toEpochMilli();
    }

    private Map<String, String> readAsMap(String path) {
        String content = this.fileSystem.readFile(path);
        if (content == null) {
            return new HashMap<String, String>();
        }
        return content.lines().map(l -> l.split(":")).collect(Collectors.toMap(s -> s[0], s -> s[1]));
    }

    private String exeFile(Activity.System.ProcessRunner runner) {
        String path = "";
        String procPidExe = String.format("/proc/%d/exe", this.processID());
        try {
            Path link = Paths.get(procPidExe, new String[0]);
            path = runner.execute("readlink", "-f", link.toString());
            int index = path.indexOf(" (deleted)");
            if (index != -1) {
                path = path.substring(0, index);
            }
        }
        catch (Exception e) {
            Logger.debug("Unable to open symbolic link " + procPidExe + ". " + e.getMessage());
        }
        return path;
    }

    private static void getMissingDetails(Map<String, String> status, String stat) {
        if (status == null || stat == null) {
            return;
        }
        int nameStart = stat.indexOf(40);
        int nameEnd = stat.indexOf(41);
        if (Util.isBlank(status.get("Name")) && nameStart > 0 && nameStart < nameEnd) {
            status.put("Name", stat.substring(nameStart + 1, nameEnd));
        }
        if (Util.isBlank(status.get("State")) && nameEnd > 0 && stat.length() > nameEnd + 2) {
            status.put("State", String.valueOf(stat.charAt(nameEnd + 2)));
        }
    }

    private long getProcessOpenFileLimit(long processId) {
        String limitsPath = String.format("/proc/%d/limits", processId);
        if (!this.fileSystem.exists(limitsPath)) {
            return -1L;
        }
        return this.fileSystem.readFile(limitsPath).lines().filter(line -> line.startsWith("Max open files")).findFirst().map(s -> Long.parseLong(s.split("\\D+")[2])).orElse(-1L);
    }

    static {
        for (ProcPidStat stat : ProcPidStat.values()) {
            LinuxOSProcess.PROC_PID_STAT_ORDERS[stat.ordinal()] = stat.getOrder() - 1;
        }
    }

    private static enum ProcPidStat {
        PPID(4),
        MINOR_FAULTS(10),
        MAJOR_FAULTS(12),
        USER_TIME(14),
        KERNEL_TIME(15),
        PRIORITY(18),
        THREAD_COUNT(20),
        START_TIME(22),
        VSZ(23),
        RSS(24);

        private final int order;

        public int getOrder() {
            return this.order;
        }

        private ProcPidStat(int order) {
            this.order = order;
        }
    }
}

