package io.intino.consul.accessor;

import com.google.gson.JsonElement;
import com.google.gson.reflect.TypeToken;
import io.intino.alexandria.Json;
import io.intino.alexandria.jms.QueueProducer;
import io.intino.alexandria.logger.Logger;

import javax.jms.*;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static io.intino.alexandria.Base64.encode;

public class ConsulAccessor implements AutoCloseable {
	private static final Type asList = new TypeToken<List<String>>() {
	}.getType();
	private static final Type asActivityList = new TypeToken<List<Activity>>() {
	}.getType();
	private static final Type asMap = new TypeToken<Map<String, String>>() {
	}.getType();
	private static final Type asBooleanMap = new TypeToken<Map<String, Boolean>>() {
	}.getType();
	private static final int DefaultTimeoutSeconds = 10;
	private static final String PATH = "service.consul.%s.request";
	private final Session session;
	private final String serverID;
	private final Map<String, Message> inbox;
	private final Map<String, Object> waiting;
	private Destination temporaryQueue;
	private MessageConsumer consumer;
	private QueueProducer producer;
	private final Object monitor = new Object();

	public ConsulAccessor(Session session, String serverID) {
		this.session = session;
		this.serverID = serverID;
		this.inbox = new HashMap<>();
		this.waiting = new HashMap<>();
	}

	public ConsulAccessor open() {
		if (producer != null) return this;
		try {
			temporaryQueue = session.createTemporaryQueue();
			consumer = session.createConsumer(temporaryQueue);
			consumer.setMessageListener(messageDispatcher());
			producer = new QueueProducer(session, String.format(PATH, serverID));
		} catch (JMSException e) {
			Logger.error(e);
		}
		return this;
	}

	public boolean isClosed() {
		return producer.isClosed();
	}

	public ActivityAccessor activity(String activity) throws AccessorException {
		List<Activity> activities = activities();
		if (activities.stream().noneMatch(a -> (a.id.equals(activity) || a.name.equals(activity)) && a.enabled))
			throw new IllegalArgumentException("Activity not found");
		Activity act = activities.stream().filter(a -> a.id.equals(activity) || a.name.equals(activity)).findFirst().orElse(null);
		return new ActivityAccessor(act);
	}

	public Info info() throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#info");
				Message response = request(message, DefaultTimeoutSeconds);
				if (response == null) return null;
				return readResponseObject(((TextMessage) response).getText(), Info.class);
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException(e.getMessage());
			}
		}
	}

	private <T> T readResponseObject(String text, Class<T> aClass) throws AccessorException {
		if (!text.contains("success")) return Json.fromJson(text, aClass);
		RequestResultWrapper result = requestResultOf(text);
		if (!result.success()) throw new AccessorException(result.response.getAsString());
		return Json.fromJson(result.response, aClass);
	}


	public Result upgrade(String version, InputStream source, File configDirectory, Map<String, String> args) throws AccessorException {
		synchronized (monitor) {
			try {
				HashMap<String, Object> map = new HashMap<>(args);
				map.put("version", version);
				addConfigurationParameters(configDirectory, map);
				final Message message = createBytesMessage(temporaryQueue, source.readAllBytes(), "container#upgrade", map);
				Message response = request(message, DefaultTimeoutSeconds * 10);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException | IOException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result upgradeActivity(String activityId, InputStream source) throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createBytesMessage(temporaryQueue, source.readAllBytes(), "container#upgradeActivity", Map.of("activity", activityId));
				Message response = request(message, DefaultTimeoutSeconds * 5);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException | IOException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result restart() throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#restart");
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result factoryReset() throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#factoryReset");
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public List<Activity> activities() throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#activities");
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) {
					RequestResultWrapper result = requestResultOf(((TextMessage) response).getText());
					if (result.success) return Json.fromJson(result.response, asActivityList);
					throw new AccessorException(result.response.getAsString());
				}
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Map<String, String> getActivityConfiguration(String activityId) throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#activityConfiguration", Map.of("activity", activityId));
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) {
					RequestResultWrapper result = requestResultOf(((TextMessage) response).getText());
					if (result.success) return Json.fromJson(result.response, asMap);
					throw new AccessorException(result.response.getAsString());
				}
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result updateActivityConfiguration(String activityId, Map<String, String> args) throws AccessorException {
		synchronized (monitor) {
			try {
				HashMap<String, Object> map = new HashMap<>(args);
				map.put("activity", activityId);
				final Message message = createMessage(temporaryQueue, "container#updateActivityConfiguration", map);
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result requestActivity(String activityName, String request, Map<String, Object> arguments) throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, activityName + "#" + request, arguments);
				Message response = request(message, DefaultTimeoutSeconds * 60);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result installActivity(String activityId, InputStream source) throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createBytesMessage(temporaryQueue, source.readAllBytes(), "container#installActivity", Map.of("activity", activityId));
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException | IOException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result enableActivity(String activityId) throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#enableActivity", Map.of("activity", activityId));
				Message response = request(message, DefaultTimeoutSeconds * 6);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result disableActivity(String activityId) throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#disableActivity", Map.of("activity", activityId));
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	public Result uninstallActivity(String activityId) throws AccessorException {
		synchronized (monitor) {
			try {
				final Message message = createMessage(temporaryQueue, "container#uninstallActivity", Map.of("activity", activityId));
				Message response = request(message, DefaultTimeoutSeconds);
				if (response != null) return resultOf((TextMessage) response);
				throw new AccessorException("Timeout exceed");
			} catch (JMSException e) {
				Logger.error(e);
				throw new AccessorException("Error in request: " + e.getMessage());
			}
		}
	}

	private static void addConfigurationParameters(File configDirectory, HashMap<String, Object> map) throws IOException {
		if (configDirectory == null) return;
		File jksFile = new File(configDirectory, "terminal.jks");
		File jtsFile = new File(configDirectory, "terminal.jts");
		File passwordFile = new File(configDirectory, "terminal.password");
		if (!jtsFile.exists() || !jtsFile.exists() || !passwordFile.exists())
			throw new IOException("Configuration files not found in directory. Expected 'terminal.jks','terminal.jts', 'terminal.password'");
		map.put("jks", encode(Files.readAllBytes(jksFile.toPath())));
		map.put("jts", encode(Files.readAllBytes(jtsFile.toPath())));
		map.put("password", Files.readString(passwordFile.toPath()));
	}

	private Message request(Message message, int timeoutSeconds) throws JMSException {
		String messageId = message.getJMSCorrelationID();
		final Object monitor = new Object();
		waiting.put(messageId, monitor);
		producer.produce(message);
		waitForResponse(monitor, messageId, timeoutSeconds);
		return inbox.remove(messageId);
	}

	private TextMessage createMessage(Destination temporaryQueue, String request) throws JMSException {
		final TextMessage message = session.createTextMessage();
		message.setJMSReplyTo(temporaryQueue);
		message.setStringProperty("request", request);
		message.setJMSCorrelationID(createRandomString());
		return message;
	}

	private Message createMessage(Destination temporaryQueue, String request, Map<String, Object> parameters) throws JMSException {
		final TextMessage message = session.createTextMessage();
		return customize(temporaryQueue, request, parameters, message);
	}

	private Message createBytesMessage(Destination temporaryQueue, byte[] content, String request, Map<String, Object> parameters) throws JMSException {
		final BytesMessage message = session.createBytesMessage();
		message.writeBytes(content);
		customize(temporaryQueue, request, parameters, message);
		return message;
	}

	private static Message customize(Destination temporaryQueue, String request, Map<String, Object> parameters, Message message) throws JMSException {
		message.setJMSReplyTo(temporaryQueue);
		message.setStringProperty("request", request);
		message.setStringProperty("parameters", Json.toJson(parameters));
		message.setJMSCorrelationID(createRandomString());
		return message;
	}

	private static String createRandomString() {
		java.util.Random random = new java.util.Random(System.currentTimeMillis());
		long randomLong = random.nextLong();
		return Long.toHexString(randomLong);
	}

	private MessageListener messageDispatcher() {
		return message -> {
			try {
				inbox.put(message.getJMSCorrelationID(), message);
				Object object = waiting.get(message.getJMSCorrelationID());
				if (object != null) synchronized (object) {
					object.notify();
				}
			} catch (JMSException e) {
				throw new RuntimeException(e);
			}
		};
	}

	private void waitForResponse(Object monitor, String messageId, long timeoutSeconds) {
		try {
			synchronized (monitor) {
				monitor.wait(timeoutSeconds * 1000);
			}
		} catch (InterruptedException ignored) {
		}
		waiting.remove(messageId);
	}

	private static Result resultOf(TextMessage response) throws JMSException {
		String text = response.getText();
		if (!text.startsWith("{")) return new Result(true, text);
		RequestResultWrapper result = requestResultOf(text);
		return new Result(result.success, result.response.getAsString());
	}

	private static RequestResultWrapper requestResultOf(String text) {
		return Json.fromJson(text, RequestResultWrapper.class);
	}

	@Override
	public void close() {
		try {
			consumer.close();
			producer.close();
		} catch (JMSException e) {
			Logger.error(e);
		}
	}

	public record Activity(String id, String name, boolean enabled) {
	}

	record RequestResultWrapper(boolean success, JsonElement response) {
	}

	public record Result(boolean success, String remarks) {
	}

	public record Info(String host, String consulVersion, Instant startDate) {
	}

	public static class AccessorException extends Exception {

		public AccessorException(String message) {
			super(message);
		}
	}

	public class ActivityAccessor {
		private final Activity activity;

		public ActivityAccessor(Activity activity) {
			this.activity = activity;
		}

		public Result execute(String operation, String observable) throws IOException, AccessorException {
			return executeOperation(operation, Map.of("observable", observable));
		}

		public Result execute(String operation, String observable, Map<String, Object> parameters) throws AccessorException {
			HashMap<String, Object> fullParameters = new HashMap<>(Map.of("observable", observable));
			if (parameters != null) fullParameters.putAll(parameters);
			return executeOperation(operation, fullParameters);
		}

		public List<String> availableOperationsFor(String observable) throws IOException, AccessorException {
			synchronized (monitor) {
				try {
					final Message message = createMessage(temporaryQueue, activity.name, Map.of("observable", observable));
					message.setBooleanProperty("availability", true);
					Message response = request(message, DefaultTimeoutSeconds);
					if (response != null) {
						RequestResultWrapper result = requestResultOf(((TextMessage) response).getText());
						if (result.success) return Json.fromJson(result.response, asList);
						throw new AccessorException(result.response.getAsString());
					}
					throw new AccessorException("Timeout exceed");
				} catch (JMSException e) {
					Logger.error(e);
					throw new AccessorException("Error in request: " + e.getMessage());
				}
			}
		}

		private Result executeOperation(String operation, Map<String, Object> parameters) throws AccessorException {
			synchronized (monitor) {
				try {
					final Message message = createMessage(temporaryQueue, activity.name + "#" + operation, parameters);
					Message response = request(message, DefaultTimeoutSeconds);
					if (response != null) return resultOf((TextMessage) response);
					throw new AccessorException("Timeout exceed");
				} catch (JMSException e) {
					Logger.error(e);
					throw new AccessorException("Error in request: " + e.getMessage());
				}
			}
		}
	}
}