package io.intino.cesar.datahub.validators;

import io.intino.ness.master.data.validation.Issue;
import io.intino.ness.master.data.validation.RecordValidator;
import io.intino.ness.master.data.validation.RecordValidator.TripletRecord;
import io.intino.ness.master.data.validation.FieldValidator;
import io.intino.ness.master.data.validation.TripletRecordStore;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Stream;

import static io.intino.ness.master.data.validation.Issue.Type.*;

public class UserValidator implements RecordValidator {

	protected final Map<String, FieldValidator> validatorsPerField = new java.util.HashMap<>(); {
		validatorsPerField.put("name", this::validateName);
		validatorsPerField.put("displayName", this::validateDisplayName);
		validatorsPerField.put("email", this::validateEmail);
	}

	@Override
	public Stream<Issue> validate(TripletRecord record, TripletRecordStore store) {
		return record.attributes().entrySet().stream()
			.map(e -> validate(e, record, store))
			.filter(Objects::nonNull)
			.reduce(Stream::concat).orElse(Stream.empty());
	}

	protected Stream<Issue> validate(Map.Entry<String, List<TripletRecord.Value>> attrib, TripletRecord record, TripletRecordStore store) {
		return validatorsPerField.getOrDefault(attrib.getKey(), FieldValidator.none()).validate(attrib.getValue(), record, store);
	}

	protected Stream<Issue> validateName(List<TripletRecord.Value> values, TripletRecord record, TripletRecordStore store) {
		if(values.isEmpty() || values.get(0).isEmpty()) return Stream.of(Issue.error(MISSING_ATTRIBUTE, "{"+record.id()+"} " + "name is not defined.").source(record.source()));
		if(values.size() > 1) return Stream.of(Issue.error(DUPLICATED_ATTRIBUTE, "{"+record.id()+"} " + "name is defined " + values.size() + " times.").source(values.get(0).source()));
		return Stream.empty();
	}

	protected Stream<Issue> validateDisplayName(List<TripletRecord.Value> values, TripletRecord record, TripletRecordStore store) {
		if(values.isEmpty() || values.get(0).isEmpty()) return Stream.of(Issue.error(MISSING_ATTRIBUTE, "{"+record.id()+"} " + "displayName is not defined.").source(record.source()));
		if(values.size() > 1) return Stream.of(Issue.error(DUPLICATED_ATTRIBUTE, "{"+record.id()+"} " + "displayName is defined " + values.size() + " times.").source(values.get(0).source()));
		return Stream.empty();
	}

	protected Stream<Issue> validateEmail(List<TripletRecord.Value> values, TripletRecord record, TripletRecordStore store) {
		if(values.isEmpty() || values.get(0).isEmpty()) return Stream.of(Issue.error(MISSING_ATTRIBUTE, "{"+record.id()+"} " + "email is not defined.").source(record.source()));
		if(values.size() > 1) return Stream.of(Issue.error(DUPLICATED_ATTRIBUTE, "{"+record.id()+"} " + "email is defined " + values.size() + " times.").source(values.get(0).source()));
		return Stream.empty();
	}

	protected boolean isListOfString(String value, TripletRecordStore store) {
		try {
			java.util.Arrays.stream(value.split(LIST_SEPARATOR)).collect(java.util.stream.Collectors.toList());
			return true;
		} catch(Exception e) {
		 	return false;
		}
	}

	protected boolean isEntity(String id, TripletRecordStore store) {
		return store.get(id) != null;
	}
}