package io.intino.cesar.datahub.validators;

import io.intino.ness.master.data.validation.Issue;
import io.intino.ness.master.data.validation.RecordValidator;
import io.intino.ness.master.data.validation.RecordValidator.TripletRecord;
import io.intino.ness.master.data.validation.FieldValidator;
import io.intino.ness.master.data.validation.TripletRecordStore;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Stream;

import static io.intino.ness.master.data.validation.Issue.Type.*;

public class TeamValidator implements RecordValidator {

	protected final Map<String, FieldValidator> validatorsPerField = new java.util.HashMap<>(); {
		validatorsPerField.put("displayName", this::validateDisplayName);
		validatorsPerField.put("channels", this::validateChannels);
	}

	@Override
	public Stream<Issue> validate(TripletRecord record, TripletRecordStore store) {
		return record.attributes().entrySet().stream()
			.map(e -> validate(e, record, store))
			.filter(Objects::nonNull)
			.reduce(Stream::concat).orElse(Stream.empty());
	}

	protected Stream<Issue> validate(Map.Entry<String, List<TripletRecord.Value>> attrib, TripletRecord record, TripletRecordStore store) {
		return validatorsPerField.getOrDefault(attrib.getKey(), FieldValidator.none()).validate(attrib.getValue(), record, store);
	}

	protected Stream<Issue> validateDisplayName(List<TripletRecord.Value> values, TripletRecord record, TripletRecordStore store) {
		if(values.isEmpty() || values.get(0).isEmpty()) return Stream.of(Issue.error(MISSING_ATTRIBUTE, "{"+record.id()+"} " + "displayName is not defined.").source(record.source()));
		if(values.size() > 1) return Stream.of(Issue.error(DUPLICATED_ATTRIBUTE, "{"+record.id()+"} " + "displayName is defined " + values.size() + " times.").source(values.get(0).source()));
		return Stream.empty();
	}

	protected Stream<Issue> validateChannels(List<TripletRecord.Value> values, TripletRecord record, TripletRecordStore store) {
		if(values.isEmpty() || values.get(0).isEmpty()) return Stream.of(Issue.error(MISSING_ATTRIBUTE, "{"+record.id()+"} " + "channels is not defined.").source(record.source()));
		if(values.size() > 1) return Stream.of(Issue.error(DUPLICATED_ATTRIBUTE, "{"+record.id()+"} " + "channels is defined " + values.size() + " times.").source(values.get(0).source()));
		return isListOfEntity(values.get(0).get(), store)
			? Stream.empty()
			: Stream.of(Issue.error(INVALID_VALUE, "{"+record.id()+"} " + "channels is an invalid List<io.intino.ness.master.model.Entity>: " + values.get(0).get()).source(values.get(0).source()));
	}

	protected boolean isListOfEntity(String value, TripletRecordStore store) {
		try {
			List<String> list = java.util.Arrays.stream(value.split(LIST_SEPARATOR)).collect(java.util.stream.Collectors.toList());
			for(String item : list)
				if(!isEntity(item, store))
					return false;
			return true;
		} catch(Exception e) {
		 	return false;
		}
	}

	protected boolean isListOfString(String value, TripletRecordStore store) {
		try {
			java.util.Arrays.stream(value.split(LIST_SEPARATOR)).collect(java.util.stream.Collectors.toList());
			return true;
		} catch(Exception e) {
		 	return false;
		}
	}

	protected boolean isEntity(String id, TripletRecordStore store) {
		return store.get(id) != null;
	}
}