package io.intino.cesar.datahub;

import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.stream.Stream;
import java.util.stream.Collectors;
import io.intino.cesar.datahub.entities.*;
import io.intino.ness.master.model.Triplet;
import io.intino.ness.master.model.TripletRecord;
import io.intino.ness.master.data.EntityListener;
import io.intino.ness.master.model.Entity;

public interface MasterTerminal {

	static MasterTerminal create() {
		return create(new MasterTerminal.Config());
	}

	static MasterTerminal create(MasterTerminal.Config config) {
		if(config.type() == Type.FullLoad) return new FullLoadMasterTerminal(config);
		if(config.type() == Type.LazyLoad) return new LazyLoadMasterTerminal(config);
		throw new IllegalArgumentException("Unknown MasterTerminal type " + config.type());
	}

	void start();
	void stop();

	void publish(Entity entity);

	MasterTerminal.Config config();

	io.intino.ness.master.serialization.MasterSerializer serializer();

	<T extends Entity> void addEntityListener(String type, EntityListener<T> listener);

	default void addUserEntityListener(EntityListener<User> listener) {
		addEntityListener("user", listener);
	}

	default void addTeamEntityListener(EntityListener<Team> listener) {
		addEntityListener("team", listener);
	}

	default void addChannelEntityListener(EntityListener<Channel> listener) {
		addEntityListener("channel", listener);
	}

	User user(String id);
	Stream<User> users();
	default List<User> userList() {return users().collect(Collectors.toList());}

	Team team(String id);
	Stream<Team> teams();
	default List<Team> teamList() {return teams().collect(Collectors.toList());}

	Channel channel(String id);
	Stream<Channel> channels();
	default List<Channel> channelList() {return channels().collect(Collectors.toList());}

	@SuppressWarnings("unchecked")
	default <T extends Entity> T asEntity(TripletRecord record) {
		switch(record.type()) {

			default: throw new IllegalArgumentException("Unknown entity type " + record.type());
		}
	}

	enum Type {
		/**
		* <p>All records will be loaded into local-memory maps on start.</p>
		*/
		FullLoad,

		/**
		* <p>Records will be loaded from the master backend on demand.</p>
		*/
		LazyLoad;

		public static Type getDefault() {return FullLoad;}
		public static Type byName(String name) {return Arrays.stream(values()).filter(e -> e.name().equalsIgnoreCase(name)).findFirst().orElse(null);}
	}

	final class Config {
		private String instanceName = "io.intino.cesar.datahub.MasterTerminal-" + UUID.randomUUID();
		private List<String> addresses = List.of("localhost:5701");
		private Type type = Type.getDefault();
		private boolean allowWriting = false;
		private boolean multithreadLoading = true;

		public Config() {}

		public Config(Config other) {
			this.instanceName = other.instanceName;
			this.addresses = other.addresses == null ? null : List.copyOf(other.addresses);
			this.type = other.type;
			this.allowWriting = other.allowWriting;
			this.multithreadLoading = other.multithreadLoading;
		}

		public String instanceName() {
			return instanceName;
		}

		public Config instanceName(String instanceName) {
			this.instanceName = instanceName;
			return this;
		}

		public List<String> addresses() {
			return addresses;
		}

		public Config addresses(List<String> addresses) {
			this.addresses = addresses;
			return this;
		}

		public Type type() {
			return type;
		}

		public Config type(Type type) {
			this.type = type;
			return this;
		}

		public boolean allowWriting() {
			return allowWriting;
		}

		public Config allowWriting(boolean allowWriting) {
			this.allowWriting = allowWriting;
			return this;
		}

		public boolean multithreadLoading() {
			return multithreadLoading;
		}

		public Config multithreadLoading(boolean multithreadLoading) {
			this.multithreadLoading = multithreadLoading;
			return this;
		}
	}
}