package io.intino.cesar.datahub;

import com.hazelcast.client.HazelcastClient;
import com.hazelcast.client.config.ClientConfig;
import com.hazelcast.client.config.ClientNetworkConfig;
import com.hazelcast.core.EntryAdapter;
import com.hazelcast.core.EntryEvent;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.map.IMap;
import io.intino.ness.master.data.EntityListener;
import io.intino.ness.master.data.EntityListener.Event;
import io.intino.ness.master.model.Entity;
import io.intino.ness.master.model.Triplet;
import io.intino.ness.master.model.TripletRecord;
import io.intino.ness.master.serialization.MasterSerializer;
import io.intino.ness.master.serialization.MasterSerializers;
import io.intino.cesar.datahub.entities.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.stream.Stream;

import static io.intino.ness.master.core.Master.*;
import static java.util.Objects.requireNonNull;

public class LazyLoadMasterTerminal implements MasterTerminal {

	private final MasterTerminal.Config config;
	private HazelcastInstance hazelcast;
	private IMap<String, String> masterMap;
	private MasterSerializer serializer;
	@SuppressWarnings("rawtypes")
	private final Map<String, List<EntityListener>> entityListeners = new HashMap<>();

	public LazyLoadMasterTerminal(MasterTerminal.Config config) {
		this.config = requireNonNull(config);
	}

	@Override
	public void start() {
		configureLogger();
		initHazelcastClient();
	}

	@Override
	public void stop() {
		hazelcast.shutdown();
	}

	@Override
	public <T extends Entity> void addEntityListener(String type, EntityListener<T> listener) {
		if(type == null) throw new NullPointerException("Type cannot be null");
		if(listener == null) throw new NullPointerException("EntryListener cannot be null");
		entityListeners.computeIfAbsent(type, k -> new ArrayList<>()).add(listener);
	}

	@Override
    public MasterSerializer serializer() {
    	return serializer;
    }

	@Override
	public MasterTerminal.Config config() {
		return new MasterTerminal.Config(config);
	}

	@Override
	public User user(String id) {
		TripletRecord record = getRecord(id);
		return record != null ? entity(User::new, id, record) : null;
	}

	@Override
	public Stream<User> users() {
		return masterMap.entrySet().stream()
				.filter(e -> e.getKey().endsWith(":user"))
				.map(e -> entity(User::new, e.getKey(), serializer.deserialize(e.getValue())));
	}

	@Override
	public Team team(String id) {
		TripletRecord record = getRecord(id);
		return record != null ? entity(Team::new, id, record) : null;
	}

	@Override
	public Stream<Team> teams() {
		return masterMap.entrySet().stream()
				.filter(e -> e.getKey().endsWith(":team"))
				.map(e -> entity(Team::new, e.getKey(), serializer.deserialize(e.getValue())));
	}

	@Override
	public Channel channel(String id) {
		TripletRecord record = getRecord(id);
		return record != null ? entity(Channel::new, id, record) : null;
	}

	@Override
	public Stream<Channel> channels() {
		return masterMap.entrySet().stream()
				.filter(e -> e.getKey().endsWith(":channel"))
				.map(e -> entity(Channel::new, e.getKey(), serializer.deserialize(e.getValue())));
	}

	private TripletRecord getRecord(String id) {
    	String serializedRecord = masterMap.get(id);
    	if(serializedRecord == null) return null;
    	return serializer.deserialize(serializedRecord);
    }

   		private <T extends Entity> T entity(BiFunction<String, MasterTerminal, T> constructor, String id, TripletRecord record) {
   			T entity = constructor.apply(id, this);
   			record.triplets().forEach(entity::add);
   			return entity;
   		}

	@Override
	public void publish(Entity entity) {
		if(!config.allowWriting()) throw new UnsupportedOperationException("This master client cannot publish because it is configured as read only");
		if(entity == null) throw new NullPointerException("Entity cannot be null");
		hazelcast.getTopic(REQUESTS_TOPIC).publish(config.instanceName() + MESSAGE_SEPARATOR + serializer().serialize(entity.asTripletRecord()));
	}

	private void initHazelcastClient() {
		ClientConfig config = new ClientConfig();
    	config.setInstanceName(this.config.instanceName());
    	config.setNetworkConfig(new ClientNetworkConfig().setAddresses(this.config.addresses()));

    	hazelcast = HazelcastClient.newHazelcastClient(config);

    	masterMap = hazelcast.getMap(MASTER_MAP_NAME);
    	IMap<String, String> metadata = hazelcast.getMap(METADATA_MAP_NAME);
    	serializer = MasterSerializers.get(metadata.get("serializer"));

    	initListeners();
	}

	private void initListeners() {
		hazelcast.getMap(MASTER_MAP_NAME).addEntryListener(new BaseEntryListener(), true);
	}

	private static void configureLogger() {
		java.util.logging.Logger rootLogger = LogManager.getLogManager().getLogger("");
		rootLogger.setLevel(Level.WARNING);
		for (Handler h : rootLogger.getHandlers()) rootLogger.removeHandler(h);
		final ConsoleHandler handler = new ConsoleHandler();
		handler.setLevel(Level.WARNING);
		handler.setFormatter(new io.intino.alexandria.logger.Formatter());
		rootLogger.setUseParentHandlers(false);
		rootLogger.addHandler(handler);
	}

	public class BaseEntryListener extends EntryAdapter<String, String> {

		@Override
		public void entryAdded(EntryEvent<String, String> event) {
			notifyEntityListeners(event, Event.Type.Create);
		}

		@Override
		public void entryUpdated(EntryEvent<String, String> event) {
			notifyEntityListeners(event, Event.Type.Update);
		}

		@Override
		public void entryRemoved(EntryEvent<String, String> event) {
			notifyEntityListeners(event, Event.Type.Remove);
		}

		@Override
		public void entryEvicted(EntryEvent<String, String> event) {
			entryRemoved(event);
		}

		@SuppressWarnings("all")
		private void notifyEntityListeners(EntryEvent<String, String> e, Event.Type type) {
			TripletRecord record = serializer().deserialize(e.getValue());
			MasterEntityEvent<?> event = new MasterEntityEvent<>(type, asEntity(record));
			List<EntityListener> listeners = entityListeners.get(record.type());
			if(listeners != null) listeners.forEach(listener -> listener.notify(event));
		}
	}

	public static class MasterEntityEvent<T extends Entity> implements Event<T> {

		private final Type type;
		private final T entity;

		private MasterEntityEvent(Type type, T entity) {
			this.type = type;
			this.entity = entity;
		}

		@Override
		public Type type() {
			return type;
		}

		@Override
		public T entity() {
			return entity;
		}
	}
}