package io.intino.cesar.datahub;

import com.hazelcast.client.HazelcastClient;
import com.hazelcast.client.config.ClientConfig;
import com.hazelcast.client.config.ClientNetworkConfig;
import com.hazelcast.core.EntryAdapter;
import com.hazelcast.core.EntryEvent;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.map.IMap;
import io.intino.alexandria.logger.Logger;
import io.intino.ness.master.data.EntityListener;
import io.intino.ness.master.data.EntityListener.Event;
import io.intino.ness.master.model.Entity;
import io.intino.ness.master.model.Triplet;
import io.intino.ness.master.model.TripletRecord;
import io.intino.ness.master.serialization.MasterSerializer;
import io.intino.ness.master.serialization.MasterSerializers;
import io.intino.cesar.datahub.entities.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.stream.Stream;

import static io.intino.ness.master.core.Master.*;
import static java.util.Objects.requireNonNull;

public class FullLoadMasterTerminal implements MasterTerminal {

	private final Map<String, User> userMap = new ConcurrentHashMap<>();
	private final Map<String, Team> teamMap = new ConcurrentHashMap<>();
	private final Map<String, Channel> channelMap = new ConcurrentHashMap<>();

	private final MasterTerminal.Config config;
	private HazelcastInstance hazelcast;
	@SuppressWarnings("rawtypes")
	private final Map<String, List<EntityListener>> entityListeners = new HashMap<>();

	public FullLoadMasterTerminal(MasterTerminal.Config config) {
		this.config = requireNonNull(config);
	}

	@Override
	public void start() {
		configureLogger();
		initHazelcastClient();
		loadData();
		initListeners();
	}

	@Override
	public void stop() {
		hazelcast.shutdown();
	}

	@Override
	public <T extends Entity> void addEntityListener(String type, EntityListener<T> listener) {
		if(type == null) throw new NullPointerException("Type cannot be null");
		if(listener == null) throw new NullPointerException("EntryListener cannot be null");
		entityListeners.computeIfAbsent(type, k -> new ArrayList<>()).add(listener);
	}

	@Override
	public MasterSerializer serializer() {
    	IMap<String, String> metadata = hazelcast.getMap(METADATA_MAP_NAME);
    	return MasterSerializers.get(metadata.get("serializer"));
    }

	@Override
	public MasterTerminal.Config config() {
		return new MasterTerminal.Config(config);
	}

	@Override
	public User user(String id) {
		return userMap.get(id);
	}

	@Override
	public Stream<User> users() {
		return userMap.values().stream();
	}

	@Override
	public Team team(String id) {
		return teamMap.get(id);
	}

	@Override
	public Stream<Team> teams() {
		return teamMap.values().stream();
	}

	@Override
	public Channel channel(String id) {
		return channelMap.get(id);
	}

	@Override
	public Stream<Channel> channels() {
		return channelMap.values().stream();
	}

	@Override
	public void publish(Entity entity) {
		if(!config.allowWriting()) throw new UnsupportedOperationException("This master client cannot publish because it is configured as read only");
		if(entity == null) throw new NullPointerException("Entity cannot be null");
		hazelcast.getTopic(REQUESTS_TOPIC).publish(config.instanceName() + MESSAGE_SEPARATOR + serializer().serialize(entity.asTripletRecord()));
	}

	private void add(TripletRecord record) {
		String enabledValue = record.getValue("enabled");
		if(enabledValue != null && !"true".equalsIgnoreCase(enabledValue)) return;

		switch(record.type()) {
			case "user": addToUser(record); break;
			case "team": addToTeam(record); break;
			case "channel": addToChannel(record); break;
		}
	}

	private void remove(String id) {
		switch(Triplet.typeOf(id)) {
			case "user": removeFromUser(id); break;
			case "team": removeFromTeam(id); break;
			case "channel": removeFromChannel(id); break;
		}
	}

	private void addToUser(TripletRecord record) {
		User entity = new User(record.id(), this);
		record.triplets().forEach(entity::add);
		userMap.put(record.id(), entity);
	}

	private void addToTeam(TripletRecord record) {
		Team entity = new Team(record.id(), this);
		record.triplets().forEach(entity::add);
		teamMap.put(record.id(), entity);
	}

	private void addToChannel(TripletRecord record) {
		Channel entity = new Channel(record.id(), this);
		record.triplets().forEach(entity::add);
		channelMap.put(record.id(), entity);
	}

	private void removeFromUser(String id) {
		userMap.remove(id);
	}

	private void removeFromTeam(String id) {
		teamMap.remove(id);
	}

	private void removeFromChannel(String id) {
		channelMap.remove(id);
	}

	private void initHazelcastClient() {
		ClientConfig config = new ClientConfig();
		config.setInstanceName(this.config.instanceName());
		config.setNetworkConfig(new ClientNetworkConfig().setAddresses(this.config.addresses()));
		hazelcast = HazelcastClient.newHazelcastClient(config);
	}

	private void initListeners() {
		hazelcast.getMap(MASTER_MAP_NAME).addEntryListener(new BaseEntryListener(), true);
	}

	private void loadData() {
		IMap<String, String> master = hazelcast.getMap(MASTER_MAP_NAME);
		MasterSerializer serializer = serializer();

		Logger.debug("Loading data from master (serializer=" + serializer.name() + ")");
		long start = System.currentTimeMillis();

		if(config.multithreadLoading())
			loadDataMultiThread(master, serializer);
		else
			loadDataSingleThread(master, serializer);

		long time = System.currentTimeMillis() - start;
		Logger.info("Data from master loaded in " + time + " ms");
	}

	private void loadDataSingleThread(IMap<String, String> master, MasterSerializer serializer) {
		master.forEach((id, serializedRecord) -> add(serializer.deserialize(serializedRecord)));
	}

	private void loadDataMultiThread(IMap<String, String> master, MasterSerializer serializer) {
		try {
			ExecutorService threadPool = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors() - 1);

			master.forEach((id, serializedRecord) -> threadPool.submit(() -> add(serializer.deserialize(serializedRecord))));

			threadPool.shutdown();
			threadPool.awaitTermination(1, TimeUnit.HOURS);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private static void configureLogger() {
		java.util.logging.Logger rootLogger = LogManager.getLogManager().getLogger("");
		rootLogger.setLevel(Level.WARNING);
		for (Handler h : rootLogger.getHandlers()) rootLogger.removeHandler(h);
		final ConsoleHandler handler = new ConsoleHandler();
		handler.setLevel(Level.WARNING);
		handler.setFormatter(new io.intino.alexandria.logger.Formatter());
		rootLogger.setUseParentHandlers(false);
		rootLogger.addHandler(handler);
	}

	public class BaseEntryListener extends EntryAdapter<String, String> {

		@Override
		public void entryAdded(EntryEvent<String, String> event) {
			addOrUpdateRecord(event.getKey(), event.getValue());
			notifyEntityListeners(event, Event.Type.Create);
		}

		@Override
		public void entryUpdated(EntryEvent<String, String> event) {
			addOrUpdateRecord(event.getKey(), event.getValue());
			notifyEntityListeners(event, Event.Type.Update);
		}

		@Override
		public void entryRemoved(EntryEvent<String, String> event) {
			remove(event.getKey());
			notifyEntityListeners(event, Event.Type.Remove);
		}

		@Override
		public void entryEvicted(EntryEvent<String, String> event) {
			entryRemoved(event);
		}

		private void addOrUpdateRecord(String id, String serializedRecord) {
			MasterSerializer serializer = serializer();
			add(serializer.deserialize(serializedRecord));
		}

		@SuppressWarnings("all")
		private void notifyEntityListeners(EntryEvent<String, String> e, Event.Type type) {
			TripletRecord record = serializer().deserialize(e.getValue());
			MasterEntityEvent<?> event = new MasterEntityEvent<>(type, asEntity(record));
			List<EntityListener> listeners = entityListeners.get(record.type());
			if(listeners != null) listeners.forEach(listener -> listener.notify(event));
		}
	}

	public static class MasterEntityEvent<T extends Entity> implements Event<T> {

		private final Type type;
		private final T entity;

		private MasterEntityEvent(Type type, T entity) {
			this.type = type;
			this.entity = entity;
		}

		@Override
		public Type type() {
			return type;
		}

		@Override
		public T entity() {
			return entity;
		}
	}
}