package io.intino.amidas.identityeditor.box.ui.displays.templates;

import io.intino.alexandria.core.Box;
import io.intino.alexandria.ui.displays.UserMessage;
import io.intino.alexandria.ui.utils.DelayerUtil;
import io.intino.alexandria.zif.grammar.Property;
import io.intino.amidas.identityeditor.box.ContextManager;
import io.intino.amidas.identityeditor.box.ui.TeamHelper;
import io.intino.amidas.identityeditor.box.ui.displays.notifiers.TeamEditorNotifier;
import io.intino.amidas.shared.Team;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

public class TeamEditor extends AbstractTeamEditor<TeamEditorNotifier, Box> {
    private Team team;
    private String condition;
    private int current = 0;
    private List<Team.Identity> identities = Collections.emptyList();
    private Consumer<Team> modifiedListener;
    private BiConsumer<Team.Identity, List<Team.Statement>> identityModifiedListener;
    private List<String> identityManagerRoles = new ArrayList<>();

    public TeamEditor(Box box) {
        super(box);
    }

    public TeamEditor team(Team team) {
        this.team = team;
        return this;
    }

    public TeamEditor identityManagerRoles(List<String> roles) {
        this.identityManagerRoles = roles;
        return this;
    }

    public TeamEditor onModify(Consumer<Team> listener) {
        this.modifiedListener = listener;
        return this;
    }

    public TeamEditor onModifyIdentity(BiConsumer<Team.Identity, List<Team.Statement>> listener) {
        this.identityModifiedListener = listener;
        return this;
    }

    public List<Property> filterProperties(List<Property> properties) {
        return properties.stream().filter(p -> TeamHelper.isRolePropertyManager(team, identityManagerRoles, user(), p)).collect(Collectors.toList());
    }

    public void filter(String condition) {
        this.condition = condition;
        identityLayer.visible(false);
        if (team != null) {
            identities = team.search(condition);
            addAll(team.search(Property.Feature, condition));
            addAll(team.search(Property.Credential, condition));
            addAll(team.search(Property.Role, condition));
        }
        else identities = Collections.emptyList();
        loadingLayer.visible(false);
        identityLayer.visible(true);
        current = 0;
    }

    private void addAll(List<Team.Identity> result) {
        List<String> ids = identities.stream().map(Team.Identity::id).collect(toList());
        result.stream().filter(i -> !ids.contains(i.id())).forEach(i -> identities.add(i));
    }

    @Override
    public void init() {
        super.init();
        initToolbar();
        initDialog();
    }

    private void initToolbar() {
        search.onChange(e -> {
            DelayerUtil.execute(this, e1 -> {
                filter(e.value());
                refresh();
            }, 1000);
        });
        search.onEnterPress(e -> filter());
        previous.onExecute(e -> previous());
        next.onExecute(e -> next());
        // TODO TEAM NO PERMITE ELIMINAR IDENTIDADES removeIdentity.onExecute(e -> removeIdentity());
    }

    private void initDialog() {
        openAddIdentity.onOpen(e -> {
            identityDialogBox.title(translate("Add identity"));
            identityDialogBox.main.dialogLayer.dialog.properties(filterProperties(team.grammar().getAll()));
            identityDialogBox.main.dialogLayer.dialog.propertyChecker(this::checkPropertyValue);
            identityDialogBox.main.dialogLayer.dialog.identity(createIdentity());
            identityDialogBox.main.dialogLayer.dialog.refresh();
        });
        openEditIdentity.onOpen(e -> {
            identityDialogBox.title(translate("Edit identity"));
            identityDialogBox.main.dialogLayer.dialog.properties(filterProperties(team.grammar().getAll()));
            identityDialogBox.main.dialogLayer.dialog.propertyChecker(this::checkPropertyValue);
            identityDialogBox.main.dialogLayer.dialog.identity(current());
            identityDialogBox.main.dialogLayer.dialog.refresh();
        });
        cancel.onExecute(e -> identityDialogBox.close());
        accept.onExecute(e -> saveIdentity());
    }

    private Boolean checkPropertyValue(Property property, String value) {
        Team.Identity identity = identityDialogBox.main.dialogLayer.dialog.identity();
        if (!property.isIdentifier() && !property.isCredential()) return true;
        List<Team.Identity> allIdentities = team.search("");
        List<String> idsValues = allIdentities.stream().filter(i -> !i.id().equals(identity.id())).map(i -> i.ids().stream().map(Team.Id::value).collect(toList())).flatMap(Collection::stream).collect(toList());
        List<String> credentialsValues = allIdentities.stream().filter(i -> !i.id().equals(identity.id())).map(i -> i.credentials().stream().map(Team.Credential::value).collect(toList())).flatMap(Collection::stream).collect(toList());
        return !idsValues.contains(value) && !credentialsValues.contains(value);
    }

    private Team.Identity createIdentity() {
        return team.createIdentity();
    }

    private void saveIdentity() {
        identityDialogBox.close();
        ContextManager.register(soul());
        notifyUser(translate("Identity saved"), UserMessage.Type.Success);
        notifySave(identityDialogBox.main.dialogLayer.dialog.identity(), identityDialogBox.main.dialogLayer.dialog.statements());
        refreshIdentity(identityDialogBox.main.dialogLayer.dialog.identity());
    }

    private void notifySaveIdentities() {
        if (modifiedListener != null) modifiedListener.accept(team);
    }

    private void notifySave(Team.Identity identity, List<Team.Statement> statements) {
        notifySaveIdentities();
        if (identityModifiedListener != null) identityModifiedListener.accept(identity, statements);
    }

    public void reloadTeam(Team team) {
        this.team = team;
        identities = team.search(condition);
        updateIdentity();
        refresh();
    }

    private void updateIdentity() {
        Team.Identity identity = identityDialogBox.main.dialogLayer.dialog.identity();
        if (identity == null) return;
        int pos = findIdentity(identity);
        this.current = pos != -1 ? pos : 0;
    }

    private int findIdentity(Team.Identity identity) {
        for (int i=0; i<identities.size(); i++) {
            if (identities.get(i).id().equals(identity.id())) return i;
        }
        return -1;
    }

    @Override
    public void refresh() {
        super.refresh();
        refreshLayers();
        if (team == null) return;
        refreshToolbar();
        refreshIdentity(current());
        refreshInfo();
    }

    private void refreshLayers() {
        identitiesFileNotFoundLayer.visible(team == null);
        teamEditorLayer.visible(team != null);
    }

    private void refreshToolbar() {
        previous.readonly(!canPrevious());
        next.readonly(!canNext());
        openAddIdentity.visible(TeamHelper.isRootIdentityManager(team, identityManagerRoles, user()));
    }

    private void refreshIdentity(Team.Identity identity) {
        identityViewer.identity(identity, filterProperties(team.grammar().getAll()));
        identityViewer.refresh();
    }

    private void refreshInfo() {
        info.value(identitiesCountMessage(identities.size()));
    }

    private Team.Identity current() {
        return identities.size() > 0 ? identities.get(current) : null;
    }

    private void previous() {
        current--;
        if (current < 0) current = 0;
        refresh();
    }

    private boolean canPrevious() {
        return identities.size() > 0 && current > 0;
    }

    private void next() {
        current++;
        if (current >= identities.size()) current = identities.size()-1;
        refresh();
    }

    private boolean canNext() {
        return identities.size() > 0 && current < identities.size()-1;
    }

    private void filter() {
        filter(condition);
        refresh();
    }

    public String identitiesCountMessage(int count) {
        if (count <= 0) return translate("No identities");
        if (count == 1) return translate("1 identity");
        return NumberFormat.getNumberInstance().format(count) + translate(" identities");
    }
}