package io.intino.amidas.identityeditor.box.ui.displays.templates;

import io.intino.alexandria.core.Box;
import io.intino.alexandria.ui.displays.UserMessage;
import io.intino.alexandria.zif.grammar.Property;
import io.intino.alexandria.zif.grammar.PropertyChecker;
import io.intino.amidas.identityeditor.box.ui.displays.DisplayHelper;
import io.intino.amidas.identityeditor.box.ui.displays.notifiers.PropertyEditorNotifier;

import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

public class PropertyEditor extends AbstractPropertyEditor<PropertyEditorNotifier, Box> {
    private Property _property;
    private String _value;
    private Consumer<String> changeListener;
    private BiFunction<Property, String, Boolean> checker;

    public PropertyEditor(Box box) {
        super(box);
    }

    public PropertyEditor property(Property property) {
        this._property = property;
        return this;
    }

    public PropertyEditor value(String value) {
        this._value = value;
        return this;
    }

    public PropertyEditor propertyChecker(BiFunction<Property, String, Boolean> checker) {
        this.checker = checker;
        return this;
    }

    public void onChange(Consumer<String> listener) {
        this.changeListener = listener;
    }

    @Override
    public void init() {
        super.init();
        initStringValue();
        initImageValue();
    }

    @Override
    public void refresh() {
        super.refresh();
        property.value(_property.label());
        refreshStringValue();
        refreshImageValue();
    }

    private void initStringValue() {
        stringValue.onChange(e -> {
            if (!check(e.value(), (v, error) -> {
                stringInvalidBlock.visible(true);
                stringInvalidValue.value(error);
            })) return;
            stringInvalidBlock.visible(false);
            changeListener.accept(e.value());
        });
    }

    private void initImageValue() {
        imageValue.onChange(e -> {
            String value = DisplayHelper.toString(e.value());
            if (!check(value, (v, error) -> {
                refreshImage(v);
                notifyUser(error, UserMessage.Type.Error);
            })) return;
            refreshImage(value);
            changeListener.accept(value);
        });
    }

    private void refreshStringValue() {
        stringType.visible(_property.type() == Property.Type.string);
        stringValue.value(_value);
        String grammar = _property.grammar();
        stringGrammarBlock.visible(grammar != null && !grammar.isEmpty());
        stringGrammarLabel.value(translate(_property.cardinality() == Property.Cardinality.multiple ? "options: " : "format: "));
        stringGrammar.value(grammar);
    }

    private void refreshImageValue() {
        boolean isImage = _property.type() == Property.Type.image;
        imageType.visible(isImage);
        if (!isImage) return;
        refreshImage(_value);
    }

    private void refreshImage(String value) {
        if (value == null || value.isEmpty()) return;
        imageValue.value(DisplayHelper.toUrl(_property, value, username()));
    }

    private boolean check(String value, BiConsumer<String, String> updater) {
        boolean result = checker.apply(_property, value);
        if (!result) {
            updater.accept(_value, translate("Already in use"));
            return false;
        }
        String grammar = _property.grammar();
        if (grammar == null || grammar.isEmpty() || value == null || value.isEmpty()) return true;
        String checkResult = new PropertyChecker(_property).check(value);
        if (checkResult == null) return true;
        updater.accept(_value, checkResult);
        return false;
    }

}