/*
 * Decompiled with CFR 0.152.
 */
package io.intino.alexandria.led;

import io.intino.alexandria.led.Led;
import io.intino.alexandria.led.LedHeader;
import io.intino.alexandria.led.LedStream;
import io.intino.alexandria.led.Schema;
import io.intino.alexandria.led.util.memory.MemoryUtils;
import io.intino.alexandria.logger.Logger;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.xerial.snappy.SnappyOutputStream;

public class LedWriter {
    private static final int DEFAULT_BUFFER_SIZE = 4096;
    private int bufferSize = 4096;
    private final OutputStream destOutputStream;
    private final File destinationFile;

    public LedWriter(File destOutputStream) {
        destOutputStream.getAbsoluteFile().getParentFile().mkdirs();
        this.destinationFile = destOutputStream;
        this.destOutputStream = this.outputStream(destOutputStream);
    }

    public LedWriter(OutputStream destOutputStream) {
        this.destOutputStream = destOutputStream;
        this.destinationFile = null;
    }

    public int bufferSize() {
        return this.bufferSize;
    }

    public LedWriter bufferSize(int bufferSize) {
        this.bufferSize = bufferSize;
        return this;
    }

    private FileOutputStream outputStream(File destination) {
        try {
            return new FileOutputStream(destination);
        }
        catch (FileNotFoundException e) {
            Logger.error((Throwable)e);
            return null;
        }
    }

    public void write(Led<? extends Schema> led) {
        this.serialize(led);
    }

    public void write(LedStream<? extends Schema> led) {
        this.serialize(led);
    }

    public <T extends Schema> void writeUncompressed(LedStream<T> ledStream) {
        if (this.destinationFile != null) {
            this.fastSerializeUncompressed(ledStream);
        } else {
            this.serializeUncompressed(ledStream);
        }
    }

    private <T extends Schema> void fastSerializeUncompressed(LedStream<T> ledStream) {
        try (FileChannel fileChannel = FileChannel.open(this.destinationFile.toPath(), StandardOpenOption.WRITE);){
            int schemaSize = ledStream.schemaSize();
            ByteBuffer outputBuffer = MemoryUtils.allocBuffer((long)this.bufferSize * (long)schemaSize);
            long destPtr = MemoryUtils.addressOf(outputBuffer);
            int offset = 0;
            while (ledStream.hasNext()) {
                Object schema = ledStream.next();
                MemoryUtils.memcpy(((Schema)schema).address() + ((Schema)schema).baseOffset(), destPtr + (long)offset, schemaSize);
                if ((offset += schemaSize) != outputBuffer.capacity()) continue;
                fileChannel.write(outputBuffer);
                outputBuffer.clear();
                offset = 0;
            }
            if (offset > 0) {
                outputBuffer.limit(offset);
                fileChannel.write(outputBuffer);
                outputBuffer.clear();
            }
            this.destOutputStream.close();
            ledStream.close();
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
    }

    private <T extends Schema> void serializeUncompressed(LedStream<T> ledStream) {
        try (OutputStream outputStream = this.destOutputStream;){
            int schemaSize = ledStream.schemaSize();
            byte[] outputBuffer = new byte[this.bufferSize * schemaSize];
            int offset = 0;
            while (ledStream.hasNext()) {
                Object schema = ledStream.next();
                MemoryUtils.memcpy(((Schema)schema).address(), ((Schema)schema).baseOffset(), outputBuffer, (long)offset, (long)schemaSize);
                if ((offset += schemaSize) != outputBuffer.length) continue;
                this.writeToOutputStream(outputStream, outputBuffer);
                offset = 0;
            }
            if (offset > 0) {
                this.writeToOutputStream(outputStream, outputBuffer, 0, offset);
            }
            ledStream.close();
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
    }

    private void serialize(Led<? extends Schema> led) {
        if (led.size() == 0L) {
            return;
        }
        ExecutorService executor = Executors.newSingleThreadExecutor();
        long size = led.size();
        int schemaSize = led.schemaSize();
        int numBatches = (int)Math.ceil((float)led.size() / (float)this.bufferSize);
        try (OutputStream fos = this.destOutputStream;){
            LedHeader header = new LedHeader();
            header.elementCount(size).elementSize(schemaSize);
            fos.write(header.toByteArray());
            try (SnappyOutputStream outputStream = new SnappyOutputStream(fos);){
                for (int i = 0; i < numBatches; ++i) {
                    int start = i * this.bufferSize;
                    int numElements = (int)Math.min((long)this.bufferSize, led.size() - (long)start);
                    byte[] outputBuffer = new byte[numElements * schemaSize];
                    for (int j = 0; j < numElements; ++j) {
                        Schema src = led.schema(j + start);
                        long offset = j * schemaSize;
                        MemoryUtils.memcpy(src.address(), src.baseOffset(), outputBuffer, offset, (long)schemaSize);
                    }
                    executor.submit(() -> this.writeToOutputStream((OutputStream)outputStream, outputBuffer));
                }
                executor.shutdown();
                executor.awaitTermination(Integer.MAX_VALUE, TimeUnit.MILLISECONDS);
            }
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
    }

    private void serialize(LedStream<? extends Schema> ledStream) {
        long elementCount = 0L;
        try (OutputStream fos = this.destOutputStream;){
            LedHeader header = new LedHeader();
            header.elementCount(-1L).elementSize(ledStream.schemaSize());
            fos.write(header.toByteArray());
            try (SnappyOutputStream outputStream = new SnappyOutputStream(fos);){
                int schemaSize = ledStream.schemaSize();
                byte[] outputBuffer = new byte[this.bufferSize * schemaSize];
                int offset = 0;
                while (ledStream.hasNext()) {
                    Object schema = ledStream.next();
                    MemoryUtils.memcpy(((Schema)schema).address(), ((Schema)schema).baseOffset(), outputBuffer, (long)offset, (long)schemaSize);
                    if ((offset += schemaSize) == outputBuffer.length) {
                        this.writeToOutputStream((OutputStream)outputStream, outputBuffer);
                        offset = 0;
                    }
                    ++elementCount;
                }
                if (offset > 0) {
                    this.writeToOutputStream((OutputStream)outputStream, outputBuffer, 0, offset);
                }
            }
            ledStream.close();
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
        if (this.destinationFile != null) {
            this.overrideHeader(elementCount, ledStream.schemaSize());
        }
    }

    private void overrideHeader(long elementCount, int schemaSize) {
        try (RandomAccessFile raFile = new RandomAccessFile(this.destinationFile, "rw");){
            raFile.writeLong(elementCount);
            raFile.writeInt(schemaSize);
        }
        catch (IOException e) {
            Logger.error((Throwable)e);
        }
    }

    private void writeToOutputStream(OutputStream outputStream, byte[] outputBuffer) {
        this.writeToOutputStream(outputStream, outputBuffer, 0, outputBuffer.length);
    }

    private void writeToOutputStream(OutputStream outputStream, byte[] outputBuffer, int offset, int size) {
        try {
            outputStream.write(outputBuffer, offset, size);
        }
        catch (IOException e) {
            Logger.error((Throwable)e);
        }
    }
}

