/*
 * Decompiled with CFR 0.152.
 */
package io.intino.alexandria.led.util.sorting;

import io.intino.alexandria.led.GenericTransaction;
import io.intino.alexandria.led.LedHeader;
import io.intino.alexandria.led.LedReader;
import io.intino.alexandria.led.LedStream;
import io.intino.alexandria.led.Transaction;
import io.intino.alexandria.led.allocators.stack.StackAllocator;
import io.intino.alexandria.led.allocators.stack.StackAllocators;
import io.intino.alexandria.led.util.memory.MemoryUtils;
import io.intino.alexandria.logger.Logger;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.LinkedList;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.stream.Stream;

public class LedExternalMergeSort {
    private static final int DEFAULT_NUM_TRANSACTIONS_IN_MEMORY = 100000;
    private final File srcFile;
    private final File destFile;
    private File chunksDirectory;
    private int numTransactionsInMemory;
    private boolean debug;
    private boolean checkChunkSorting;
    private ByteBuffer primaryBuffer;
    private ByteBuffer secondaryBuffer;
    private LedHeader ledHeader;
    private Queue<Path> chunks;
    private boolean deleteChunkDirOnExit;
    private double startTime = 0.0;

    public LedExternalMergeSort(File srcFile, File destFile) {
        this.srcFile = srcFile;
        this.destFile = destFile;
        destFile.getParentFile().mkdirs();
        File defaultChunkDir = new File(srcFile.getParentFile(), Thread.currentThread().getName() + "_Chunks_Dir_" + System.nanoTime());
        this.chunksDirectory(defaultChunkDir);
        this.numTransactionsInMemory(100000);
    }

    public int maxMemoryUsed(int transactionSize) {
        return this.numTransactionsInMemory * transactionSize + this.numTransactionsInMemory * 32;
    }

    public int numTransactionsInMemory() {
        return this.numTransactionsInMemory;
    }

    public LedExternalMergeSort numTransactionsInMemory(int numTransactionsInMemory) {
        this.numTransactionsInMemory = Math.max(1000, numTransactionsInMemory);
        return this;
    }

    public File chunksDirectory() {
        return this.chunksDirectory;
    }

    public LedExternalMergeSort chunksDirectory(File chunksDirectory) {
        this.chunksDirectory = chunksDirectory;
        chunksDirectory.mkdirs();
        return this;
    }

    public boolean debug() {
        return this.debug;
    }

    public LedExternalMergeSort debug(boolean debug) {
        this.debug = debug;
        return this;
    }

    public boolean checkChunkSorting() {
        return this.checkChunkSorting;
    }

    public LedExternalMergeSort checkChunkSorting(boolean checkChunkSorting) {
        this.checkChunkSorting = checkChunkSorting;
        return this;
    }

    private int transactionSize() {
        return this.ledHeader.elementSize();
    }

    private long transactionsCount() {
        return this.ledHeader.elementCount();
    }

    public boolean deleteChunkDirOnExit() {
        return this.deleteChunkDirOnExit;
    }

    public LedExternalMergeSort deleteChunkDirOnExit(boolean deleteChunkDirOnExit) {
        this.deleteChunkDirOnExit = deleteChunkDirOnExit;
        return this;
    }

    public void sort() {
        try {
            if (this.debug) {
                Logger.info((String)("Starting external merge sort: " + this.srcFile + " -> " + this.destFile + "(using " + this.numTransactionsInMemory + " transactions in memory)..."));
            }
            this.startTime = System.currentTimeMillis();
            this.createSortedChunks();
            if (this.transactionsCount() == 0L) {
                this.handleEmptyFile();
            } else {
                this.mergeSortedChunks();
            }
            if (this.debug) {
                Logger.info((String)("External merge sort finished after " + this.time()));
            }
        }
        catch (Throwable e) {
            Logger.error((String)("Failed to merge sort " + this.srcFile + " to " + this.destFile + ": " + e.getMessage()), (Throwable)e);
        }
        finally {
            this.freeBuffers();
            this.chunks = null;
            this.ledHeader = null;
            if (this.deleteChunkDirOnExit) {
                this.chunksDirectory.delete();
                this.chunksDirectory.deleteOnExit();
            }
        }
    }

    private String time() {
        double time = ((double)System.currentTimeMillis() - this.startTime) / 1000.0;
        return time + " minutes";
    }

    private void handleEmptyFile() {
        LedStream.empty().serialize(this.destFile);
    }

    private void createSortedChunks() {
        try (FileChannel fileChannel = FileChannel.open(this.srcFile.toPath(), StandardOpenOption.READ);){
            long fileSize = fileChannel.size();
            String srcFileName = this.srcFile.getName();
            this.ledHeader = LedHeader.from(fileChannel);
            if ((this.ledHeader == null || this.ledHeader.elementCount() == 0L) && this.debug) {
                Logger.info((String)("File " + srcFileName + " is empty. Not merge sorting."));
            }
            int transactionSize = this.ledHeader.elementSize();
            int bufferSize = this.numTransactionsInMemory * transactionSize / 2;
            if (this.debug) {
                Logger.info((String)("Buffer Size = " + (double)bufferSize / 1024.0 / 1024.0 + " MB"));
            }
            this.allocateBuffers(bufferSize);
            this.chunks = new LinkedList<Path>();
            StackAllocator<GenericTransaction> allocator = this.createAllocator(this.primaryBuffer);
            PriorityQueue<GenericTransaction> priorityQueue = new PriorityQueue<GenericTransaction>(this.numTransactionsInMemory);
            Path chunkDir = this.chunksDirectory.toPath();
            int i = 0;
            while (fileChannel.position() < fileSize) {
                Path chunk = Files.createFile(chunkDir.resolve("Chunk[" + i + "]_" + srcFileName), new FileAttribute[0]);
                this.clearBuffers();
                int bytesRead = fileChannel.read(this.primaryBuffer);
                this.clearBuffers();
                this.writeSortedChunk(chunk, transactionSize, allocator, priorityQueue, bytesRead);
                if (this.checkChunkSorting) {
                    this.checkSorting(chunk, transactionSize);
                }
                this.chunks.add(chunk);
                priorityQueue.clear();
                allocator.clear();
                ++i;
            }
            priorityQueue.clear();
            allocator.clear();
        }
        catch (Throwable e) {
            Logger.error((Throwable)e);
        }
    }

    private void clearBuffers() {
        this.primaryBuffer.clear();
        this.secondaryBuffer.clear();
    }

    private void allocateBuffers(int bufferSize) {
        this.primaryBuffer = MemoryUtils.allocBuffer(bufferSize);
        this.secondaryBuffer = MemoryUtils.allocBuffer(bufferSize);
    }

    private StackAllocator<GenericTransaction> createAllocator(ByteBuffer buffer) {
        return StackAllocators.newManaged(this.transactionSize(), buffer, GenericTransaction::new);
    }

    private void writeSortedChunk(Path chunk, int transactionSize, StackAllocator<GenericTransaction> allocator, PriorityQueue<GenericTransaction> priorityQueue, int elementsRead) {
        try (FileChannel fileChannel = FileChannel.open(chunk, StandardOpenOption.WRITE);){
            this.sortChunk(allocator, priorityQueue, transactionSize, elementsRead);
            fileChannel.write(this.secondaryBuffer);
        }
        catch (IOException e) {
            Logger.error((Throwable)e);
        }
    }

    private void sortChunk(StackAllocator<GenericTransaction> allocator, PriorityQueue<GenericTransaction> priorityQueue, int transactionSize, int bytesRead) {
        for (int i = 0; i < bytesRead; i += transactionSize) {
            priorityQueue.add(allocator.malloc());
        }
        int limit = priorityQueue.size() * transactionSize;
        long tempBufferPtr = MemoryUtils.addressOf(this.secondaryBuffer);
        long offset = 0L;
        while (!priorityQueue.isEmpty()) {
            GenericTransaction transaction = priorityQueue.poll();
            MemoryUtils.memcpy(transaction.address() + transaction.baseOffset(), tempBufferPtr + offset, transactionSize);
            offset += (long)transactionSize;
        }
        this.secondaryBuffer.position(0).limit(limit);
    }

    private void mergeSortedChunks() throws IOException {
        if (this.chunks.size() == 1) {
            this.writeFinalChunkToDestFile(this.destFile, this.chunks.remove(), this.ledHeader);
            this.removeDirectory(this.chunksDirectory);
            return;
        }
        int transactionSize = this.transactionSize();
        int count = 0;
        Path chunksDir = this.chunksDirectory.toPath();
        while (this.chunks.size() > 2) {
            Path chunk1 = this.chunks.remove();
            Path chunk2 = this.chunks.remove();
            Path mergedChunk = Files.createFile(chunksDir.resolve("Chunk_" + count + "_merged_with_" + (count + 1) + ".led.tmp"), new FileAttribute[0]);
            this.merge(chunk1, chunk2, mergedChunk, transactionSize);
            if (this.checkChunkSorting) {
                this.checkSorting(mergedChunk, transactionSize);
            }
            this.deleteChunks(chunk1, chunk2);
            this.chunks.add(mergedChunk);
            count += 2;
        }
        if (this.debug) {
            Logger.info((String)("Running final merge..." + this.time()));
        }
        this.freeBuffers();
        if (this.chunks.size() == 2) {
            this.doFinalMergeAndWriteToDestFile(this.destFile, this.chunks.remove(), this.chunks.remove(), this.ledHeader);
        } else if (this.chunks.size() == 1) {
            this.writeFinalChunkToDestFile(this.destFile, this.chunks.remove(), this.ledHeader);
        } else {
            throw new IllegalStateException("Number of chunks is neither 1 or 2.");
        }
    }

    private void freeBuffers() {
        MemoryUtils.free(this.primaryBuffer);
        MemoryUtils.free(this.secondaryBuffer);
    }

    private void merge(Path chunk1, Path chunk2, Path mergedChunk, int transactionSize) {
        LedStream.merged(Stream.of(this.readChunk(chunk1, transactionSize), this.readChunk(chunk2, transactionSize))).serializeUncompressed(mergedChunk.toFile());
    }

    private void removeDirectory(File dir) {
        dir.delete();
        dir.deleteOnExit();
    }

    private void deleteChunks(Path ... chunks) {
        for (Path chunk : chunks) {
            File chunkFile = chunk.toFile();
            chunkFile.delete();
            chunkFile.deleteOnExit();
        }
    }

    private void checkSorting(Path chunk, int transactionSize) {
        try (LedStream<GenericTransaction> ledStream = this.readChunk(chunk, transactionSize);){
            long previousId = Long.MIN_VALUE;
            while (ledStream.hasNext()) {
                Object transaction = ledStream.next();
                long id = Transaction.idOf((Transaction)transaction);
                if (id < previousId) {
                    throw new IllegalStateException(chunk + " is unsorted: " + previousId + " > " + Transaction.idOf((Transaction)transaction));
                }
                previousId = id;
            }
        }
        catch (IllegalStateException e) {
            throw e;
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
    }

    private void doFinalMergeAndWriteToDestFile(File destFile, Path chunk1, Path chunk2, LedHeader ledHeader) {
        int elementSize = ledHeader.elementSize();
        LedStream.merged(Stream.of(this.readChunk(chunk1, elementSize), this.readChunk(chunk2, elementSize))).serialize(destFile);
        this.deleteChunks(chunk1, chunk2);
        this.deleteDir(chunk1.getParent());
    }

    private void writeFinalChunkToDestFile(File destFile, Path chunk, LedHeader ledHeader) {
        int elementSize = ledHeader.elementSize();
        this.readChunk(chunk, elementSize).serialize(destFile);
        this.deleteChunks(chunk);
    }

    private void deleteDir(Path dir) {
        File dirFile = dir.toFile();
        dirFile.delete();
        dirFile.deleteOnExit();
    }

    private LedStream<GenericTransaction> readChunk(Path chunk, int elementSize) {
        return new LedReader(chunk.toFile()).readUncompressed(elementSize, GenericTransaction::new);
    }
}

