/*
 * Decompiled with CFR 0.152.
 */
package io.intino.alexandria.led;

import io.intino.alexandria.led.Led;
import io.intino.alexandria.led.LedHeader;
import io.intino.alexandria.led.LedStream;
import io.intino.alexandria.led.Transaction;
import io.intino.alexandria.led.util.memory.MemoryUtils;
import io.intino.alexandria.logger.Logger;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.xerial.snappy.SnappyOutputStream;

public class LedWriter {
    private static final int DEFAULT_BUFFER_SIZE = 4096;
    private int bufferSize = 4096;
    private final OutputStream destOutputStream;
    private final File destinationFile;

    public LedWriter(File destOutputStream) {
        destOutputStream.getAbsoluteFile().getParentFile().mkdirs();
        this.destinationFile = destOutputStream;
        this.destOutputStream = this.outputStream(destOutputStream);
    }

    public LedWriter(OutputStream destOutputStream) {
        this.destOutputStream = destOutputStream;
        this.destinationFile = null;
    }

    public int bufferSize() {
        return this.bufferSize;
    }

    public LedWriter bufferSize(int bufferSize) {
        this.bufferSize = bufferSize;
        return this;
    }

    private FileOutputStream outputStream(File destination) {
        try {
            return new FileOutputStream(destination);
        }
        catch (FileNotFoundException e) {
            Logger.error((Throwable)e);
            return null;
        }
    }

    public void write(Led<? extends Transaction> led) {
        this.serialize(led);
    }

    public void write(LedStream<? extends Transaction> led) {
        this.serialize(led);
    }

    public <T extends Transaction> void writeUncompressed(LedStream<T> ledStream) {
        if (this.destinationFile != null) {
            this.fastSerializeUncompressed(ledStream);
        } else {
            this.serializeUncompressed(ledStream);
        }
    }

    private <T extends Transaction> void fastSerializeUncompressed(LedStream<T> ledStream) {
        try (FileChannel fileChannel = FileChannel.open(this.destinationFile.toPath(), StandardOpenOption.WRITE);){
            int transactionSize = ledStream.transactionSize();
            ByteBuffer outputBuffer = MemoryUtils.allocBuffer((long)this.bufferSize * (long)transactionSize);
            long destPtr = MemoryUtils.addressOf(outputBuffer);
            int offset = 0;
            while (ledStream.hasNext()) {
                Object transaction = ledStream.next();
                MemoryUtils.memcpy(((Transaction)transaction).address() + ((Transaction)transaction).baseOffset(), destPtr + (long)offset, transactionSize);
                if ((offset += transactionSize) != outputBuffer.capacity()) continue;
                fileChannel.write(outputBuffer);
                outputBuffer.clear();
                offset = 0;
            }
            if (offset > 0) {
                outputBuffer.limit(offset);
                fileChannel.write(outputBuffer);
                outputBuffer.clear();
            }
            this.destOutputStream.close();
            ledStream.close();
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
    }

    private <T extends Transaction> void serializeUncompressed(LedStream<T> ledStream) {
        try (OutputStream outputStream = this.destOutputStream;){
            int transactionSize = ledStream.transactionSize();
            byte[] outputBuffer = new byte[this.bufferSize * transactionSize];
            int offset = 0;
            while (ledStream.hasNext()) {
                Object schema = ledStream.next();
                MemoryUtils.memcpy(((Transaction)schema).address(), ((Transaction)schema).baseOffset(), outputBuffer, (long)offset, (long)transactionSize);
                if ((offset += transactionSize) != outputBuffer.length) continue;
                this.writeToOutputStream(outputStream, outputBuffer);
                offset = 0;
            }
            if (offset > 0) {
                this.writeToOutputStream(outputStream, outputBuffer, 0, offset);
            }
            ledStream.close();
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
    }

    private void serialize(Led<? extends Transaction> led) {
        if (led.size() == 0L) {
            return;
        }
        ExecutorService executor = Executors.newSingleThreadExecutor();
        long size = led.size();
        int transactionSize = led.transactionSize();
        int numBatches = (int)Math.ceil((float)led.size() / (float)this.bufferSize);
        try (OutputStream fos = this.destOutputStream;){
            LedHeader header = new LedHeader();
            header.elementCount(size).elementSize(transactionSize);
            fos.write(header.toByteArray());
            try (SnappyOutputStream outputStream = new SnappyOutputStream(fos);){
                for (int i = 0; i < numBatches; ++i) {
                    int start = i * this.bufferSize;
                    int numElements = (int)Math.min((long)this.bufferSize, led.size() - (long)start);
                    byte[] outputBuffer = new byte[numElements * transactionSize];
                    for (int j = 0; j < numElements; ++j) {
                        Transaction src = led.transaction(j + start);
                        long offset = j * transactionSize;
                        MemoryUtils.memcpy(src.address(), src.baseOffset(), outputBuffer, offset, (long)transactionSize);
                    }
                    executor.submit(() -> this.writeToOutputStream((OutputStream)outputStream, outputBuffer));
                }
                executor.shutdown();
                executor.awaitTermination(Integer.MAX_VALUE, TimeUnit.MILLISECONDS);
            }
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
    }

    private void serialize(LedStream<? extends Transaction> ledStream) {
        long elementCount = 0L;
        try (OutputStream fos = this.destOutputStream;){
            LedHeader header = new LedHeader();
            header.elementCount(-1L).elementSize(ledStream.transactionSize());
            fos.write(header.toByteArray());
            try (SnappyOutputStream outputStream = new SnappyOutputStream(fos);){
                int transactionSize = ledStream.transactionSize();
                byte[] outputBuffer = new byte[this.bufferSize * transactionSize];
                int offset = 0;
                while (ledStream.hasNext()) {
                    Object schema = ledStream.next();
                    MemoryUtils.memcpy(((Transaction)schema).address(), ((Transaction)schema).baseOffset(), outputBuffer, (long)offset, (long)transactionSize);
                    if ((offset += transactionSize) == outputBuffer.length) {
                        this.writeToOutputStream((OutputStream)outputStream, outputBuffer);
                        offset = 0;
                    }
                    ++elementCount;
                }
                if (offset > 0) {
                    this.writeToOutputStream((OutputStream)outputStream, outputBuffer, 0, offset);
                }
            }
            ledStream.close();
        }
        catch (Exception e) {
            Logger.error((Throwable)e);
        }
        if (this.destinationFile != null) {
            this.overrideHeader(elementCount, ledStream.transactionSize());
        }
    }

    private void overrideHeader(long elementCount, int transactionSize) {
        try (RandomAccessFile raFile = new RandomAccessFile(this.destinationFile, "rw");){
            raFile.writeLong(elementCount);
            raFile.writeInt(transactionSize);
        }
        catch (IOException e) {
            Logger.error((Throwable)e);
        }
    }

    private void writeToOutputStream(OutputStream outputStream, byte[] outputBuffer) {
        this.writeToOutputStream(outputStream, outputBuffer, 0, outputBuffer.length);
    }

    private void writeToOutputStream(OutputStream outputStream, byte[] outputBuffer, int offset, int size) {
        try {
            outputStream.write(outputBuffer, offset, size);
        }
        catch (IOException e) {
            Logger.error((Throwable)e);
        }
    }
}

