/*
 * Decompiled with CFR 0.152.
 */
package io.intino.alexandria.fsm;

import io.intino.alexandria.fsm.FileSessionManager;
import io.intino.alexandria.fsm.Mailbox;
import io.intino.alexandria.fsm.TimePeriod;
import io.intino.alexandria.logger.Logger;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

public class LockFile {
    public static final String LOCK_EXTENSION = ".lock";
    private final String fileSessionManagerId;
    private final Mailbox inputMailbox;
    private final File file;
    private static final DateTimeFormatter Formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd hh:mm:ss.SS");

    public static List<File> getLockFilesOf(Mailbox mailbox) {
        File[] files = mailbox.root().listFiles(f -> f.getName().endsWith(LOCK_EXTENSION));
        return files == null ? Collections.emptyList() : Arrays.asList(files);
    }

    public LockFile(FileSessionManager fsm) {
        this.fileSessionManagerId = Objects.requireNonNull(fsm).id();
        this.inputMailbox = fsm.inputMailbox();
        this.file = new File(this.inputMailbox.root(), this.fileSessionManagerId + LOCK_EXTENSION);
    }

    public void delete() {
        this.file.delete();
    }

    public boolean exists() {
        return this.file.exists();
    }

    public void validate() {
        this.ensureNotExistOtherLockFile();
        this.createIfNotExists();
    }

    public void write(String message) {
        try {
            if (!this.file.exists()) {
                return;
            }
            Files.writeString(this.file.toPath(), (CharSequence)(this.format(LocalDateTime.now()) + message + "\n"), new OpenOption[0]);
        }
        catch (IOException e) {
            Logger.error((String)("Could not write message to " + this.fileSessionManagerId + " lock file: " + e.getMessage()), (Throwable)e);
        }
    }

    private String format(LocalDateTime ts) {
        return "[" + Formatter.format(ts) + "]: ";
    }

    public boolean waitForRelease(TimePeriod timeout) {
        if (timeout == null) {
            Logger.warn((String)("FSM " + this.fileSessionManagerId + " will wait indefinitely until lock from " + this.inputMailbox.root().getPath() + " is released. This could result in an endless wait loop."));
        }
        Instant startTime = Instant.now();
        Instant waitLimit = timeout == null ? null : Instant.now().plus(timeout.amount(), timeout.temporalUnit());
        int elapsedHours = 0;
        List<File> lockFiles;
        while (!(lockFiles = LockFile.getLockFilesOf(this.inputMailbox)).isEmpty()) {
            if (lockFiles.size() == 1 && lockFiles.get(0).equals(this.file)) {
                return true;
            }
            if (this.hasReachedTimeout(waitLimit)) {
                return false;
            }
            this.sleep(5000);
            int hoursSinceStart = (int)Math.abs(ChronoUnit.HOURS.between(startTime, Instant.now()));
            if (hoursSinceStart <= elapsedHours) continue;
            Logger.warn((String)("FSM " + this.fileSessionManagerId + " is still waiting for lock of " + this.inputMailbox.root().getPath() + " to be released (elapsed hours=" + hoursSinceStart + ")..."));
            elapsedHours = hoursSinceStart;
        }
        return true;
    }

    private void createIfNotExists() {
        if (!this.exists()) {
            try {
                this.inputMailbox.mkdirs();
                this.file.createNewFile();
                this.file.deleteOnExit();
            }
            catch (IOException e) {
                Logger.error((String)("Failed to create " + this.fileSessionManagerId + " lock file: " + e.getMessage()), (Throwable)e);
            }
        }
    }

    public String fileSessionManagerId() {
        return this.fileSessionManagerId;
    }

    public Mailbox inputMailbox() {
        return this.inputMailbox;
    }

    public File file() {
        return this.file;
    }

    private void ensureNotExistOtherLockFile() {
        File[] files = this.inputMailbox.root().listFiles(f -> f.getName().endsWith(LOCK_EXTENSION));
        if (files == null || files.length == 0) {
            return;
        }
        if (files.length > 1) {
            throw new LockFileException("There are multiple lock files in " + this.inputMailbox.root().getPath());
        }
        if (!files[0].equals(this.file)) {
            throw new LockFileException("There is already a lock file of other FSM in " + this.inputMailbox.root().getPath() + ": " + files[0]);
        }
    }

    private boolean hasReachedTimeout(Instant timeout) {
        return timeout != null && Instant.now().isBefore(timeout);
    }

    private void sleep(int millis) {
        try {
            Thread.sleep(millis);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public static class LockFileException
    extends IllegalStateException {
        public LockFileException(String s) {
            super(s);
        }
    }
}

