package io.intino.tara.builder.semantic;

import io.intino.tara.Checker;
import io.intino.tara.Resolver;
import io.intino.tara.language.model.Mogram;
import io.intino.tara.language.model.MogramRoot;
import io.intino.tara.language.semantics.errorcollector.SemanticException;
import io.intino.tara.language.semantics.errorcollector.SemanticFatalException;

import java.util.ArrayList;
import java.util.List;

public class SemanticAnalyzer {
	private final MogramRoot root;
	private final Resolver resolver;
	private Checker checker;
	private List<SemanticException> notifications;

	public SemanticAnalyzer(MogramRoot root) {
		this.root = root;
		resolver = new Resolver(root.language());
		checker = new Checker(root.language());
		notifications = new ArrayList<>();
	}

	public void analyze() throws SemanticFatalException {
		resolveTypes(root);
		checkNode(root);
		if (!notifications.isEmpty()) throw new SemanticFatalException(notifications);
	}

	private void resolveTypes(Mogram mogram) {
		mogram.components().forEach(this::resolveNode);
	}

	private void check(Mogram mogram) {
		mogram.components().forEach(this::checkNode);
	}

	private void resolveNode(Mogram mogram) {
		resolver.resolve(mogram);
		if (!mogram.isReference()) resolveTypes(mogram);
	}

	private void checkNode(Mogram mogram) {
		try {
			checker.check(mogram);
			if (!mogram.isReference()) check(mogram);
		} catch (SemanticFatalException e) {
			notifications.addAll(e.exceptions());
			if (!hasFatal(e.exceptions()) && !mogram.isReference()) check(mogram);
		}
	}

	private boolean hasFatal(List<SemanticException> exceptions) {
		for (SemanticException exception : exceptions)
			if (exception.isFatal()) return true;
		return false;
	}
}